<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use App\Models\FoodSearch;
use Carbon\Carbon;

class FoodService
{

    public function getAuthenticationToken()
    {
        $existingToken = FoodSearch::latest()->first();

        // Verify token expiration
        if ($existingToken && $existingToken->expires_at > now()) {
            return $existingToken->authentication_token;
        }

        // Request a new token
        $response = Http::asForm()->post('https://oauth.fatsecret.com/connect/token', [
            'grant_type' => 'client_credentials',
            'client_id' => env('FATSECRET_CLIENT_ID'),
            'client_secret' => env('FATSECRET_CLIENT_SECRET'),
            'scope' => 'premier',
        ]);

        if ($response->successful()) {
            $data = $response->json();
            $token = $data['access_token'];
            $expiresAt = Carbon::now()->addSeconds($data['expires_in']);

            // Store the token in the database
            FoodSearch::create([
                'authentication_token' => $token,
                'expires_at' => $expiresAt,
            ]);

            return $token;
        }

        // Handle the case where token retrieval fails
        throw new \Exception('Failed to retrieve authentication token.');
    }


    public function searchFood($searchExpression, $region = 'USA', $format = 'json')
    {
        $token = $this->getAuthenticationToken();

        if (!$token) {
            return response()->json(['error' => 'Failed to fetch authentication token'], 500);
        }

        // Debugging log
        // \Log::info('Using token for Food Search API:', ['token' => $token]);
        // return response()->json(['success' => 'Using token for Food Search API', 'token' => $token], 200);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $token,
        ])->get('https://platform.fatsecret.com/rest/foods/search/v3', [
            'search_expression' => $searchExpression,
            'region' => $region,
            'format' => $format,
        ]);




        if ($response->successful()) {
            return $response->json();
        }

        return response()->json(['error' => 'Food search request failed', 'details' => $response->json()], 500);
    }


}
