<?php

namespace App\Http\Controllers; 
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use App\Helpers\NotificationHelper; 
use App\Services\FoodService;


 
use App\Models\User;
use App\Models\Coach;
use App\Models\Clients;
use App\Models\Session;
use App\Models\Subscription;
use App\Models\paymentMethod;
use App\Models\Invoice;
use App\Models\checkInDetails;
use App\Models\CheckInPictures;
use App\Models\macroPlans;
use App\Models\exercises;
use App\Models\workoutPlans;
use App\Models\workoutDays;
use App\Models\repsSets;

use App\Models\Meals;
use App\Models\MealItems;
use App\Models\dietDays;
use App\Models\dietPlans;
use App\Models\clientPreferences;

use App\Models\faq;
use App\Models\Complaints;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Carbon;

use Illuminate\Support\Facades\Storage;
use Stripe\StripeClient;
use Stripe\Event;
use Stripe\Webhook;
use Stripe\Exception\SignatureVerificationException;
use Illuminate\Support\Facades\Http;

use Stripe\Stripe;



use Illuminate\Support\Facades\Crypt;







class APIController extends Controller
{
    const STATUS_SUCCESS = 200;
    const STATUS_BAD_REQUEST = 400;
    const STATUS_UNAUTHORIZED= 401;
    const STATUS_UNAUTHORIZED_EMAIL= 403;
    const STATUS_NOT_FOUND = 404;
    const STATUS_CONFLICT = 409;
    const STATUS_STRIPE_NOT_FOUND = 422;
    const STATUS_INTERNAL_SERVER_ERROR = 500;
    // Define more status codes as needed

    protected $foodService;


    public function __construct(FoodService $foodService)
    {
        $this->foodService = $foodService;
    }



    private function errorResponse($message, $statusCode)
    {
        return response(['success' => 0, 'message' => $message], $statusCode);
    }

    private function successResponse($message, $extraData = [])
    {
        $response = array_merge(['success' => 1, 'message' => $message], $extraData);
        return response($response, self::STATUS_SUCCESS);
    }

    public function checkCoachUsername(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'username' => 'required|string|max:255',
        ]);

        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $coach = Coach::where('username', $request->username)->first();
        if ($coach) {
            return $this->successResponse("Coach found", ["alreadyExists" => 1]);
        } else {
            return $this->successResponse("Coach doesn't exist", ["alreadyExists" => 0]);
        }
    }

    public function registerAdmin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'username' => 'required|string|max:255|unique:coaches', 
            'full_name' => 'required|string|max:255',             
            'phone' => 'required|string|max:255',  
            'email' => 'required|string|email|max:255|unique:coaches',                       
            'password' => 'required|string|max:255',
            'city' => 'required|string|max:255',    
            'postal_code' => 'required|string|max:255',    
            'country' => 'required|string|max:255',
        ]);

        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $request['user_role'] = "admin";

        $request['password'] = Hash::make($request['password']);
        $insertedID = DB::table('coaches')->insertGetId($request->toArray());
        if ($insertedID) {
            $message = "Admin registered successfully";
            return $this->successResponse($message, ['admin_id' => $insertedID]);
        } else {
            return $this->errorResponse('Failed to register coach', self::STATUS_INTERNAL_SERVER_ERROR);
        }

    }



    // public function registerCoach(Request $request)
    // {
    //     $validator = Validator::make($request->all(), [
    //         'username' => 'required|string|max:255|unique:coaches', 
    //         'full_name' => 'required|string|max:255',             
    //         'phone' => 'required|string|max:255',  
    //         'email' => 'required|string|email|max:255|unique:coaches',                       
    //         'password' => 'required|string|max:255',
    //         'city' => 'required|string|max:255',    
    //         'postal_code' => 'required|string|max:255',    
    //         'country' => 'required|string|max:255',
    //         // 'bio' => 'string|max:255',
    //         // 'intro' => 'string|max:255',
    //         // 'profile_picture' => 'string|max:255',
    //         // 'stripe_id' => 'string|string|max:255', 
    //         // 'user_role' => 'string|in:admin,coach',
    //         // 'is_email_verified' => 'string|in:true,false,1,0', 
    //     ]);

    //     if ($validator->fails())
    //     {
    //         return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
    //     }
        


    //     // Initialize Stripe with your secret key
    //     $stripe = new StripeClient(env('STRIPE_SECRET_KEY'));

    //     // Create customer in Stripe
    //     $stripeCustomer = $stripe->customers->create([
    //         'email' => $request['email'],
    //         'name' => $request['full_name'],
    //         'phone' => $request['phone'],
    //         'address' => [
    //             'city' => $request['city'],
    //             'country' => $request['country'],
    //             'postal_code' => $request['postal_code'],
    //         ],
    //     ]);

    //     $request['stripe_id']  = $stripeCustomer->id;





    //     // Convert 'true'/'false' to 1/0
    //     $request['is_email_verified'] = $request['is_email_verified'] === 'true' ? 1 : 0;

    //     $request['password'] = Hash::make($request['password']);
    //     $insertedID = DB::table('coaches')->insertGetId($request->toArray());
    //     if ($insertedID) {
    //         $coach = Coach::where('email', $request->email)->first();

    //         // Generate a random 4-digit code
    //         $verificationCode = random_int(1000, 9999);

    //         // Calculate expiration time (e.g., 10 minutes from now)
    //         $expirationTime = Carbon::now()->addMinutes(10);

    //         // Send the verification code to the email
    //         try {
    //             Mail::send('emails.verification_code', [
    //                 'email' => $request->email,
    //                 'verificationCode' => $verificationCode,
    //             ], function ($message) use ($request) {
    //                 $message->to($request->email)
    //                         ->subject("Let's Verify Your Email for The Coaching App");
    //             });

    //             // Store the verification code and expiration time in the database
    //             $coach->code_for_email_verification = $verificationCode;
    //             $coach->code_for_email_verification_expires_at = $expirationTime;
    //             $coach->save();

    //         } catch (\Exception $e) {
    //             return $this->successResponse('Coach registered successfully, but failed to send verification code to your email: ' . $request->email, ['error' => $e->getMessage()]);
    //         }

    //         $message = "Coach registered successfully, verification code sent to your email: " . $request->email . ", your code will be expired within 10 minutes from now.";
    //         return $this->successResponse($message, ['coach_id' => $insertedID]);
    //     } else {
    //         return $this->errorResponse('Failed to register coach', self::STATUS_INTERNAL_SERVER_ERROR);
    //     }
    // }

    public function registerCoach(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'username' => 'required|string|max:255|unique:coaches', 
            'full_name' => 'required|string|max:255',             
            'phone' => 'required|string|max:255',  
            'email' => 'required|string|email|max:255|unique:coaches',                       
            'password' => 'required|string|max:255',
            'city' => 'required|string|max:255',    
            'postal_code' => 'required|string|max:255',    
            'country' => 'required|string|max:255',
        ]);

        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Initialize Stripe with your secret key
        $stripe = new StripeClient(env('STRIPE_SECRET_KEY'));

        // Create customer in Stripe
        $stripeCustomer = $stripe->customers->create([
            'email' => $request['email'],
            'name' => $request['full_name'],
            'phone' => $request['phone'],
            'address' => [
                'city' => $request['city'],
                'country' => $request['country'],
                'postal_code' => $request['postal_code'],
            ],
        ]);

        $request['stripe_id']  = $stripeCustomer->id;

        // Set default profile picture if not provided
        $request['profile_picture'] = '/storage/images/profile_picture_default.png';

        $request['is_email_verified'] = $request['is_email_verified'] === 'true' ? 1 : 0;
        $request['password'] = Hash::make($request['password']);
        $insertedID = DB::table('coaches')->insertGetId($request->toArray());

        if ($insertedID) {
            $coach = Coach::where('email', $request->email)->first();

            // Generate a random 4-digit code
            $verificationCode = random_int(1000, 9999);

            // Calculate expiration time (e.g., 10 minutes from now)
            $expirationTime = Carbon::now()->addMinutes(10);

            // Send the verification code to the email
            try {
                Mail::send('emails.verification_code', [
                    'email' => $request->email,
                    'verificationCode' => $verificationCode,
                ], function ($message) use ($request) {
                    $message->to($request->email)
                            ->subject("Let's Verify Your Email for The Coaching App");
                });

                // Store the verification code and expiration time in the database
                $coach->code_for_email_verification = $verificationCode;
                $coach->code_for_email_verification_expires_at = $expirationTime;
                $coach->save();

            } catch (\Exception $e) {
                return $this->successResponse('Coach registered successfully, but failed to send verification code to your email: ' . $request->email, ['error' => $e->getMessage()]);
            }

            $message = "Coach registered successfully, verification code sent to your email: " . $request->email . ", your code will be expired within 10 minutes from now.";
            return $this->successResponse($message, ['coach_id' => $insertedID]);
        } else {
            return $this->errorResponse('Failed to register coach', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }

    // public function updateCoach(Request $request)
    // {
    //     $validator = Validator::make($request->all(), [
    //         'coach_id' => 'required|string|max:255|exists:coaches,coach_id',                              
    //         'full_name' => 'string|max:255',                                    
    //         'phone' => 'string|max:255',                        
    //         'bio' => 'string|max:255',             
    //         'intro' => 'file|mimes:mp4,mov,avi', // Adjust the file type and size as needed
    //         'profile_picture' => 'image|mimes:jpeg,png,jpg,gif', // Adjust the file type and size as needed            
    //         'stripe_id' => 'string|max:255', 
    //         'city' => 'string|max:255',                                    
    //         'postal_code' => 'string|max:255',                        
    //         'country' => 'string|max:255', 
    //     ]);
        
    //     if ($validator->fails())
    //     {
    //         return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
    //     }

    //     $coach = Coach::find($request->coach_id);
    //     if (!$coach) {
    //         return $this->errorResponse('Coach not found.', self::STATUS_NOT_FOUND);
    //     }

    //     // Delete previous video and photo files if they exist
    //     if ($coach->intro) {
    //         Storage::delete(str_replace('/storage', 'public', $coach->intro));
    //     }
    //     if ($coach->profile_picture) {
    //         Storage::delete(str_replace('/storage', 'public', $coach->profile_picture));
    //     }

    //     // Handle file uploads
    //     if($request->hasFile('intro')) {
    //         $filename = $request->file('intro')->getClientOriginalName(); // get the file name
    //         // $getfilenamewitoutext = pathinfo($filename, PATHINFO_FILENAME); // get the file name without extension
    //         $getfilenamewitoutext = "intro"; // Give new file_name
    //         $getfileExtension = $request->file('intro')->getClientOriginalExtension(); // get the file extension
    //         $createnewFileName = str_replace(' ','_', $getfilenamewitoutext).'_'.time().'.'.$getfileExtension; // create new random file name
    //         $img_path = $request->file('intro')->storeAs('public/videos', $createnewFileName); // store and get the image path
    //         // Here the img_path is public/videos.$createnewFileName;  so I don't want that to be stored in the db, instead:
    //         $coach->intro = '/storage/videos/' . $createnewFileName; 
    //     }
    //     if($request->hasFile('profile_picture')) {
    //         $filename = $request->file('profile_picture')->getClientOriginalName(); // get the file name
    //         // $getfilenamewitoutext = pathinfo($filename, PATHINFO_FILENAME); // get the file name without extension
    //         $getfilenamewitoutext = "profile_picture"; // Give new file_name
    //         $getfileExtension = $request->file('profile_picture')->getClientOriginalExtension(); // get the file extension
    //         $createnewFileName = str_replace(' ','_', $getfilenamewitoutext).'_'.time().'.'.$getfileExtension; // create new random file name
    //         $img_path = $request->file('profile_picture')->storeAs('public/images', $createnewFileName); // store and get the image path
    //         // Here the img_path is public/images.$createnewFileName;  so I don't want that to be stored in the db, instead:
    //         $coach->profile_picture = '/storage/images/' . $createnewFileName; 
    //     }

    //     // Update other fields
    //     $coach->full_name = $request->input('full_name', $coach->full_name);
    //     $coach->phone = $request->input('phone', $coach->phone);
    //     $coach->bio = $request->input('bio', $coach->bio);
    //     $coach->stripe_id = $request->input('stripe_id', $coach->stripe_id);
    //     $coach->city = $request->input('city', $coach->city);
    //     $coach->postal_code = $request->input('postal_code', $coach->postal_code);
    //     $coach->country = $request->input('country', $coach->country);

    //     // Save changes
    //     $coach->save();
    //     return $this->successResponse('Coach updated successfully', ['coach_id' => $request->coach_id]);
    // }

    public function updateCoach(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255|exists:coaches,coach_id',                              
            'full_name' => 'string|max:255',                                     
            'phone' => 'string|max:255',                        
            'bio' => 'string|max:255',             
            'intro' => 'file|mimes:mp4,mov,avi', // Adjust the file type and size as needed
            'profile_picture' => 'image|mimes:jpeg,png,jpg,gif', // Adjust the file type and size as needed            
            'stripe_id' => 'string|max:255', 
            'city' => 'string|max:255',                                    
            'postal_code' => 'string|max:255',                        
            'country' => 'string|max:255', 
        ]);
        
        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $coach = Coach::find($request->coach_id);
        if (!$coach) {
            return $this->errorResponse('Coach not found.', self::STATUS_NOT_FOUND);
        }

        // Delete previous video and photo files if they exist
        if ($coach->intro) {
            Storage::delete(str_replace('/storage', 'public', $coach->intro));
        }
        if ($coach->profile_picture && $coach->profile_picture !== '/storage/images/profile_picture_default.png') {
            Storage::delete(str_replace('/storage', 'public', $coach->profile_picture));
        }

        // Handle file uploads
        if($request->hasFile('intro')) {
            $filename = $request->file('intro')->getClientOriginalName();
            $getfilenamewitoutext = "intro";
            $getfileExtension = $request->file('intro')->getClientOriginalExtension();
            $createnewFileName = str_replace(' ','_', $getfilenamewitoutext).'_'.time().'.'.$getfileExtension;
            $img_path = $request->file('intro')->storeAs('public/videos', $createnewFileName);
            $coach->intro = '/storage/videos/' . $createnewFileName; 
        }
        if($request->hasFile('profile_picture')) {
            $filename = $request->file('profile_picture')->getClientOriginalName();
            $getfilenamewitoutext = "profile_picture";
            $getfileExtension = $request->file('profile_picture')->getClientOriginalExtension();
            $createnewFileName = str_replace(' ','_', $getfilenamewitoutext).'_'.time().'.'.$getfileExtension;
            $img_path = $request->file('profile_picture')->storeAs('public/images', $createnewFileName);
            $coach->profile_picture = '/storage/images/' . $createnewFileName; 
        }

        // Set default profile picture if not updated
        if (empty($coach->profile_picture)) {
            $coach->profile_picture = '/storage/images/profile_picture_default.png';
        }

        // Update other fields
        $coach->full_name = $request->input('full_name', $coach->full_name);
        $coach->phone = $request->input('phone', $coach->phone);
        $coach->bio = $request->input('bio', $coach->bio);
        $coach->stripe_id = $request->input('stripe_id', $coach->stripe_id);
        $coach->city = $request->input('city', $coach->city);
        $coach->postal_code = $request->input('postal_code', $coach->postal_code);
        $coach->country = $request->input('country', $coach->country);

        // Save changes
        $coach->save();
        return $this->successResponse('Coach updated successfully', ['coach_id' => $request->coach_id]);
    }




    public function verifyCoachEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|exists:coaches,email',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Find the coach by email
        $coach = Coach::where('email', $request->email)->first();

        // Generate a random 4-digit code
        $verificationCode = random_int(1000, 9999);

        // Calculate expiration time (e.g., 10 minutes from now)
        $expirationTime = Carbon::now()->addMinutes(10);

        // Send the verification code to the email
        try {
            Mail::send('emails.verification_code', [
                'email' => $request->email,
                'verificationCode' => $verificationCode,
            ], function ($message) use ($request) {
                $message->to($request->email)
                        ->subject("Let's Verify Your Email for The Coaching App");
            });

            // Store the verification code and expiration time in the database
            $coach->code_for_email_verification = $verificationCode;
            $coach->code_for_email_verification_expires_at = $expirationTime;
            $coach->save();

        } catch (\Exception $e) {
            // Log the detailed error
            \Log::error('Mail send error: ' . $e->getMessage());

            // Return an error response
            return $this->errorResponse('Failed to send verification code.', self::STATUS_INTERNAL_SERVER_ERROR);
        }

        // Respond with the code for front-end verification
        return $this->successResponse('Verification code sent, your code will be expired within 10 minutes from now.');
    }

    public function verifyVerificationCodeforEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|exists:coaches,email',
            'code' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $coach = Coach::where('email', $request->email)->first();
        if (!$coach) {
            return $this->errorResponse('Coach not found', self::STATUS_NOT_FOUND);
        }

        if ($coach->code_for_email_verification != $request->code) {
            return $this->errorResponse('Invalid verification code', self::STATUS_BAD_REQUEST);
        }

        // Check if the verification code has expired
        if ($coach->code_for_email_verification_expires_at < Carbon::now()) {
            return $this->errorResponse('Verification code has expired', self::STATUS_CONFLICT);
        }

        // Verification successful, 
        // Clear the verification code and expiration time from the database
        $coach->is_email_verified = 1;        
        $coach->code_for_email_verification = null;
        $coach->code_for_email_verification_expires_at = null;
        $coach->save();




        // Extract device_info from User-Agent header
        $device_info = $request->header('User-Agent');
        
        // IDs for concatenation
        $coach_id = $coach->coach_id;  

        // Concatenating IDs and encrypting
        $idsPayload = "{$coach_id}-coach";
        $encryptedIds = encrypt($idsPayload);

        // Option A: Allow Multiple Sessions per Coach
        // Create a new session record
        $session = new Session([
            // 'device_info' => $request->device_info,
            'device_info' => $device_info, // Use extracted device_info from headers
            'authentication_token' => $encryptedIds,
            'user_id' => $coach_id,
            'user_type' => "coach"
        ]);
        $session->save();

        // Retrieve the full session from the database to include all fields
        $fullSession = Session::find($session->session_id);

        $response = [
            "success" => 1,
            'session' => $fullSession->toArray(), // Returning the full session data
        ];


        return response($response, self::STATUS_SUCCESS);


        return $this->successResponse('Verification successful');
    }




    public function verifyCoachForgetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|exists:coaches,email',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $coach = Coach::where('email', $request->email)->first();

        // Generate a random 4-digit code
        $verificationCode = random_int(1000, 9999);

        // Calculate expiration time (e.g., 10 minutes from now)
        $expirationTime = Carbon::now()->addMinutes(10);

        // Send the verification code to the email
        try {

            Mail::send('emails.forgot_password_verification', [
                'email' => $request->email,
                'verificationCode' => $verificationCode,
            ], function ($message) use ($request) {
                $message->to($request->email)
                        ->subject("Let's Verify Your Email to Reset Your Password");
            });


            // Store the verification code and expiration time in the database
            $coach->code_for_forgot_password_verification = $verificationCode;
            $coach->code_for_forgot_password_verification_expires_at = $expirationTime;
            $coach->save();

        } catch (\Exception $e) {
            // Log the detailed error
            \Log::error('Mail send error: ' . $e->getMessage());
            // Optionally, return the error message for debugging (not recommended for production)
            return $this->errorResponse('Failed to send verification code.', self::STATUS_INTERNAL_SERVER_ERROR);
        }

        // Respond with the code for front-end verification
        return $this->successResponse('Verification code sent, your code will be expired within 10 minutes from now.');
    }


    public function verifyVerificationCodeforForgetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|exists:coaches,email',
            'code' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $coach = Coach::where('email', $request->email)->first();

        if (!$coach || $coach->code_for_forgot_password_verification != $request->code) {
            return $this->errorResponse('Invalid verification code', self::STATUS_BAD_REQUEST);
        }

        // Use Carbon to ensure consistent time comparison
        $currentTime = Carbon::now();
        $expirationTime = Carbon::parse($coach->code_for_forgot_password_verification_expires_at);

        // Check if the verification code has expired
        if ($currentTime->gt($expirationTime)) {
            return $this->errorResponse('Verification code has expired', self::STATUS_BAD_REQUEST);
        }

        // Verification successful
        // Clear the verification code and expiration time from the database
        $coach->code_for_forgot_password_verification = null;
        $coach->code_for_forgot_password_verification_expires_at = null;
        $coach->save();

        return $this->successResponse('Verification successful');
    }

    public function createCoachPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|exists:coaches,email', 
            'new_password' => 'required|string|max:255',                         
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Fetch the coach from the database
        $coach = Coach::where('email', $request->email)->first();

        // Proceed with adding new password
        $coach->password = Hash::make($request->new_password);
        $addSuccess = $coach->save(); // Add the created password

        if ($addSuccess) {
            return $this->successResponse('New password created successfully');
        } else {
            return $this->errorResponse('Failed to create new password', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }

    public function updateCoachPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255|exists:coaches,coach_id', 
            'current_password' => 'required|string|max:255',                         
            'new_password' => 'required|string|max:255',                         
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Fetch the coach from the database
        $coach = Coach::where('coach_id', $request->coach_id)->first();

        // Check if the current_password matches the password in the database
        if (!Hash::check($request->current_password, $coach->password)) {
            // If the passwords don't match, return an error response
            return $this->errorResponse('Current password is incorrect', self::STATUS_BAD_REQUEST);
        }

        // If the password matches, proceed to update with the new password
        $coach->password = Hash::make($request->new_password);
        $updateSuccess = $coach->save(); // Save the updated password

        if ($updateSuccess) {
            return $this->successResponse('Password updated successfully');
        } else {
            return $this->errorResponse('Failed to update password', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }

    public function fetchCoachDetails(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255|exists:coaches,coach_id', 
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Fetch the coach without eager loading
        $coach = Coach::where('coach_id', $request->coach_id)->first();

        if ($coach) {

            // Append base URL to intro and profile_picture if they exist
            if ($coach->intro) {
                $coach->intro = url($coach->intro);
            }

            if ($coach->profile_picture) {
                $coach->profile_picture = url($coach->profile_picture);
            }  

            // Retrieve the latest Payment Method
            $latestPaymentMethod = PaymentMethod::where('coach_id', $coach->coach_id)
                ->orderBy('updated_at', 'desc')
                ->first();

            $response = [
                "success" => 1, 
                "coach" => $coach, // Or $coach->toArray() if you prefer it as an array
                "payment_method" => $latestPaymentMethod ? $latestPaymentMethod->toArray() : null,
            ];
            return response($response, self::STATUS_SUCCESS);
        } else {
            return $this->errorResponse('Failed to fetch coach details', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }


    public function coachSignIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email_username' => 'required|string|max:255', 
            'password' => 'required|string|max:255', 
            // 'device_info' => 'required|string|max:255', 
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Extract device_info from User-Agent header
        $device_info = $request->header('User-Agent');

        // Attempt to retrieve the coach by email or username
        $coach = Coach::where('email', $request->email_username)
                      ->orWhere('username', $request->email_username)->first();

        // Check if the coach exists in the database
        if (!$coach) {
            return $this->errorResponse('No account found with provided credentials', self::STATUS_NOT_FOUND);
        }


        // Verify the password is correct
        if (!Hash::check($request->password, $coach->password)) {
            return $this->errorResponse('Incorrect password', self::STATUS_UNAUTHORIZED);
        }

        // IDs for concatenation
        $coach_id = $coach->coach_id;  


        // Check admin here:
        if($coach->user_role === "admin"){
            // Concatenating IDs and encrypting
            $idsPayload = "{$coach_id}-admin";
            $encryptedIds = encrypt($idsPayload);
            $user_type  = "admin";

            $latestSubscription = [];
            $latestInvoice = [];

        }

        else{

            // Check if the email is verified
            if (!$coach->is_email_verified) {
                return $this->errorResponse('Email is not verified', self::STATUS_UNAUTHORIZED_EMAIL);
            }

            // Check if the Stripe ID is found
            if (!$coach->stripe_id) {
                return $this->errorResponse('Stripe ID is not found', self::STATUS_STRIPE_NOT_FOUND);
            }

            // Concatenating IDs and encrypting
            $idsPayload = "{$coach_id}-coach";
            $encryptedIds = encrypt($idsPayload);
            $user_type  = "coach";

            // Retrieve the latest subscription
            $latestSubscription = Subscription::where('coach_id', $coach->coach_id)
                ->orderBy('updated_at', 'desc')
                ->first();

            // Retrieve the latest invoice
            $latestInvoice = Invoice::where('coach_id', $coach->coach_id)
                ->orderBy('updated_at', 'desc')
                ->first();

        }


        // Option A: Allow Multiple Sessions per Coach
        // Create a new session record
        $session = new Session([
            // 'device_info' => $request->device_info,
            'device_info' => $device_info, // Use extracted device_info from headers
            'authentication_token' => $encryptedIds,
            'user_id' => $coach->coach_id,
            'user_type' => $user_type
        ]);
        $session->save();

        // Retrieve the full session from the database to include all fields
        $fullSession = Session::find($session->session_id);



        $response = [
            "success" => 1,
            'session' => $fullSession->toArray(), // Returning the full session data
            "coach" => $coach->toArray(), // Sending as array for consistency
            "latest_subscription" => $latestSubscription ? $latestSubscription->toArray() : null,
            "latest_invoice" => $latestInvoice ? $latestInvoice->toArray() : null,
        ];


        return response($response, self::STATUS_SUCCESS);
    }



    public function updateStripeID(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'stripe_id' => 'required|string|max:255', 
            'user_email' => 'required|string|email|max:255|exists:coaches,email',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Fetch the coach from the database
        $coach = Coach::where('email', $request->user_email)->first();
        if (!$coach) {
            return $this->errorResponse('Coach not found.', self::STATUS_NOT_FOUND);
        }

        // If the password matches, proceed to update with the new password
        $coach->stripe_id = $request->input('stripe_id', $coach->stripe_id);
        $updateSuccess = $coach->save(); // Save the updated password

        if ($updateSuccess) {
            return $this->successResponse('Stripe ID updated successfully');
        } else {
            return $this->errorResponse('Failed to update password', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }




    public function fetchClientDetails(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255|exists:clients,coach_id', 
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Fetch the client without eager loading
        $client = Clients::where('coach_id', $request->coach_id)->first();

        if ($client) {


            $response = [
                "success" => 1, 
                "client" => $client, 
            ];
            // return response($response, self::STATUS_SUCCESS);
            return $this->successResponse('Client details fetched successfully', $response);

        } else {
            return $this->errorResponse('Failed to fetch client details', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }



    public function getClientSecret(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255|exists:coaches,email', 
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Initialize Stripe with your secret key
        $stripe = new StripeClient(env('STRIPE_SECRET_KEY'));

        try {
            // Retrieve coach_id based on email
            $coach = Coach::where('email', $request['email'])->firstOrFail();
            $coachId = $coach->coach_id;
            $stripeCustomerId = $coach->stripe_id;

            // Create subscription in Stripe
            $subscription = $stripe->subscriptions->create([
                'customer' => $stripeCustomerId,
                'items' => [[
                    'price' => env('STRIPE_PRICE_ID'), // Retrieve price_id from environment or configuration
                ]],
                'payment_behavior' => 'default_incomplete',
                'payment_settings' => ['save_default_payment_method' => 'on_subscription'],
                'expand' => ['latest_invoice.payment_intent'],
            ]);


            // Store subscription details in subscriptions table
            Subscription::create([
                'stripe_subscription_id' => $subscription->id,
                'customer' => $stripeCustomerId,
                'coach_id' => $coachId,
                'status' => $subscription->status,
            ]);

            // Extract invoice details from latest_invoice object
            $latestInvoice = $subscription->latest_invoice;
            $invoiceAmountPaid = $latestInvoice->amount_paid / 100; // Convert to dollars

            // Store invoice details in invoices table
            Invoice::create([
                'stripe_invoice_id' => $latestInvoice->id,
                'customer' => $stripeCustomerId,
                'coach_id' => $coachId,
                'amount_paid' => $invoiceAmountPaid,
                'client_secret' => $latestInvoice->payment_intent->client_secret,
                'status' => $latestInvoice->status,
            ]);

            // Return coach_id, stripe_subscription_id, and client_secret to frontend
            return response()->json([
                'coach_id' => $coachId,
                'stripe_subscription_id' => $subscription->id,
                'client_secret' => $latestInvoice->payment_intent->client_secret,
            ], 200);

        } catch (\Stripe\Exception\ApiErrorException $e) {
            // Handle Stripe API errors
            return response()->json(['error' => $e->getMessage()], 500);
        } catch (\Exception $e) {
            // Handle other errors, e.g., coach not found
            return response()->json(['error' => $e->getMessage()], 404);
        }


    }



    public function handleWebhook(Request $request)
    {
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $secret = env('STRIPE_WEBHOOK_SECRET'); 

        try {
            $event = Webhook::constructEvent($payload, $sigHeader, $secret);

            // Handle different event types
            switch ($event->type) {
                case 'customer.subscription.deleted':
                case 'customer.subscription.updated':
                case 'customer.subscription.paused':
                case 'customer.subscription.resumed':
                    $this->handleSubscriptionEvent($event);
                    break;

                case 'payment_method.attached':
                    $this->handlePaymentMethodAttached($event);
                    break;

                case 'invoice.paid':
                case 'invoice.payment_action_required':
                case 'invoice.payment_failed':
                    $this->handleInvoiceEvent($event);
                    break;

                case 'invoice.created':
                    $this->handleInvoiceCreated($event);
                    break;

                default:
                    // Handle other events or log unexpected events
                    break;
            }

            return response()->json(['status' => 'success']);
        } catch (SignatureVerificationException $e) {
            return response()->json(['error' => 'Webhook signature verification failed.'], 400);
        } catch (\UnexpectedValueException $e) {
            return response()->json(['error' => 'Webhook payload parsing failed.'], 400);
        }
    }

    protected function handleSubscriptionEvent($event)
    {
        // Handle subscription events: update subscription status
        $subscription = $event->data->object;
        $customer = $subscription->customer;
        $stripeSubscriptionId = $subscription->id;

        // Update subscription status in your database
        Subscription::where('customer', $customer)
                    ->where('stripe_subscription_id', $stripeSubscriptionId)
                    ->update(['status' => $subscription->status]);
    }

    protected function handlePaymentMethodAttached($event)
    {
        // Handle payment method attached event: store new payment method details
        $paymentMethod = $event->data->object;
        $customer = $paymentMethod->customer;
        $coachId = Coach::where('stripe_id', $customer)->value('id');

        // Store payment method details in payment_methods table
        PaymentMethod::create([
            'stripe_payment_method_id' => $paymentMethod->id,
            'customer' => $customer,
            'coach_id' => $coachId,
            'type' => $paymentMethod->type,
            'card_brand' => $paymentMethod->card->brand,
            'last_4_digits' => $paymentMethod->card->last4,
            'exp_month' => $paymentMethod->card->exp_month,
            'exp_year' => $paymentMethod->card->exp_year,
            'country' => $paymentMethod->billing_details->address->country,
            'postal_code' => $paymentMethod->billing_details->address->postal_code,
        ]);
    }

    protected function handleInvoiceEvent($event)
    {
        // Handle invoice events: update invoice status
        $invoice = $event->data->object;
        $customer = $invoice->customer;
        $stripeInvoiceId = $invoice->id;

        // Update invoice status in your database
        Invoice::where('customer', $customer)
               ->where('stripe_invoice_id', $stripeInvoiceId)
               ->update(['status' => $invoice->status]);
    }


    protected function handleInvoiceCreated($event)
    {
        // Handle invoice created event: store new invoice details
        $invoice = $event->data->object;

        // Store invoice details in invoices table
        Invoice::create([
            'stripe_invoice_id' => $invoice->id,
            'customer' => $invoice->customer,
            'coach_id' => Coach::where('stripe_id', $invoice->customer)->value('id'),
            'amount_paid' => $invoice->amount_paid / 100, // Convert to dollars
            'status' => $invoice->status,
        ]);

        $stripe = new StripeClient(env('STRIPE_SECRET_KEY'));
        // Call Stripe API to update invoice to auto-advance
        $stripe->invoices->update($invoice->id, ['auto_advance' => true]);

    }


    public function CheckSubscriptionStatus(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        try {
            // Retrieve the coach ID based on the authentication token
            // $session = Session::where('authentication_token', $request->authentication_token)->firstOrFail();
            $session = Session::where('authentication_token', $token)->firstOrFail();
            if ($session){
                $coachId = $session->user_id;
                $coach = Coach::where('coach_id', $coachId)->firstOrFail();
                if ($coach){

                    // Check if the Stripe ID is found
                    if (!$coach->stripe_id) {
                            return response()->json([
                                "status" => self::STATUS_STRIPE_NOT_FOUND,
                                "coach" => $coach->toArray(), // Sending as array for consistency
                            ], self::STATUS_STRIPE_NOT_FOUND);
                    }
                    


                    // Retrieve the latest subscription
                    $latestSubscription = Subscription::where('coach_id', $coachId)
                        ->orderBy('created_at', 'desc')
                        ->first();

                    if ($latestSubscription) {
                        return response()->json([
                            "status" => self::STATUS_SUCCESS,
                            "subscription_status" => $latestSubscription->status,
                            "coach" => $coach->toArray(), // Sending as array for consistency
                        ], self::STATUS_SUCCESS);

                    } else {
                        return response()->json([
                            "status" => self::STATUS_NOT_FOUND,
                            "subscription_status" => Null,
                            "coach" => $coach->toArray(), // Sending as array for consistency
                        ], self::STATUS_NOT_FOUND);
                    }
                }
                else{
                    return response()->json([
                        "status" => self::STATUS_STRIPE_NOT_FOUND,
                        "message" => "coach  not found", // Sending as array for consistency
                    ], self::STATUS_STRIPE_NOT_FOUND); 
                }
            }
            else{
                return response()->json([
                    "status" => self::STATUS_STRIPE_NOT_FOUND,
                    "message" => "authentication token not found", // Sending as array for consistency
                ], self::STATUS_STRIPE_NOT_FOUND); 
            }

        } catch (\Exception $e) {
            // Handle other unexpected errors
            return response()->json([
                "status" => self::STATUS_INTERNAL_SERVER_ERROR,
                "subscription_status" => Null,
            ], self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }




    public function getCoachSubscriptions(Request $request)
    {
        $request->validate([
            'coach_id' => 'required|string|max:255|exists:coaches,coach_id', 
        ]);

        $coach = Coach::find($request->coach_id);

        if (!$coach) {
            return response()->json(['error' => 'Coach not found'], 404);
        }

        $stripeId = $coach->stripe_id;

        if (!$stripeId) {
            return response()->json(['error' => 'Stripe ID not found for this coach'], 404);
        }

        // Set your secret key. Remember to switch to your live secret key in production!
        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

       try {
            // Fetch subscriptions from Stripe
            $subscriptions = \Stripe\Subscription::all([
                'customer' => $coach->stripe_id,
                'status' => 'all'
            ]);

            return response()->json($subscriptions);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }



    }




    public function getClients(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coachId = $session->user_id;

        // Set default pagination values
        $page = $request->input('page', 1); // default to 1 if not provided
        $limit = $request->input('limit', 10); // default to 10 if not provided

        // Calculate the offset for the query
        $offset = ($page - 1) * $limit;

        // Fetch clients sorted by descending order against the coach_id with pagination
        $clientsQuery = Clients::where('coach_id', $coachId)
            ->with(['checkInDetails' => function ($query) {
                $query->orderBy('created_at', 'desc')->with('checkInPictures');
            }])
            ->orderBy('created_at', 'desc');

        // Get total count of clients
        $totalClients = $clientsQuery->count();

        // Get paginated clients
        $clients = $clientsQuery->skip($offset)->take($limit)->get();

        // Calculate total pages
        $totalPages = ceil($totalClients / $limit);

        // Return the response as a JSON array with pagination metadata
        return response()->json([
            'status' => 'success',
            'meta' => [
                'page' => $page,
                'pages' => $totalPages,
                'limit' => $limit,
                'total' => $totalClients,
            ],
            'clients' => $clients,
            // 'coachId' => $coachId
        ], 200);
    }




    public function getCheckinDetails(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coachId = $session->user_id;

        // Fetch check_in_details based on coach_id with nested check_in_pictures and client details
        $checkInDetails = checkInDetails::where('coach_id', $coachId)
            ->with(['checkInPictures' => function ($query) {
                $query->orderBy('created_at', 'desc');
            }, 'client'])  // Include the client details
            ->orderBy('created_at', 'desc')
            ->get();

        // Return the response as a JSON array
        return response()->json([
            'status' => 'success',
            'check_in_details' => $checkInDetails
        ], 200);
    }



    public function assignMacro(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
            'details' => 'required|string', // Ensure details is a string (JSON)
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Decode the JSON string into an array
        $details = json_decode($request->details, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return response()->json(['error' => 'Invalid JSON format in details'], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                         ->where('coach_id', $coach_id)
                         ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        DB::beginTransaction();

        try {
            // Check if the pair (coach_id, client_assigned_id) with specific types exist in the macro_plans table
            $existingMacros = macroPlans::where('coach_id', $coach_id)
                                        ->where('client_assigned_id', $request->client_id)
                                        ->whereIn('type', ['Low', 'Medium', 'High'])
                                        ->get();

            // If more than 3 records exist, something is wrong (extra safety check)
            if ($existingMacros->count() >= 3) {
                return response()->json(['error' => 'Maximum of three macro plans (Low, Medium, High) already exist for this client'], 400);
            }

            // Flag to determine if new macros are created
            $newMacrosCreated = true;

            if ($existingMacros->isNotEmpty()) {
                // Setting to false since we are updating
                $newMacrosCreated = false; 

                // Delete existing data for the pair (coach_id, client_assigned_id) with the same types
                macroPlans::where('coach_id', $coach_id)
                          ->where('client_assigned_id', $request->client_id)
                          ->whereIn('type', ['Low', 'Medium', 'High'])
                          ->delete();
            }

            // Iterate over the details array to create new Macro Plans
            foreach ($details as $macroDetail) {
                // Validate the type before creating a new record
                if (!in_array($macroDetail['type'], ['Low', 'Medium', 'High'])) {
                    return response()->json(['error' => 'Invalid macro plan type. Allowed types are: Low, Medium, High'], 400);
                }

                $newMacro = macroPlans::create([
                    'coach_id' => $coach_id,
                    'client_assigned_id' => $request->client_id,
                    'title' => $macroDetail['title'],
                    'protein' => $macroDetail['protein'],
                    'carbs' => $macroDetail['carbs'],
                    'fats' => $macroDetail['fats'],
                    'type' => $macroDetail['type'],
                    'note' => $macroDetail['note'],
                ]);
            }

            // Update the client's is_macro_assigned field to true
            $client->update(['is_macro_assigned' => true]);

            // Create a notification based on whether it's a new macro assignment or an update
            $notificationType = $newMacrosCreated ? 1 : 2; // 1: "Macro Plan Assigned", 2: "Macro Plan Updated"
            NotificationHelper::createNotification($request->client_id, $coach_id, $notificationType, $newMacro->macro_plan_id);

            DB::commit();

            return response()->json(['success' => 'Macro plan assigned successfully'], 200);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Failed to assign macro plan', 'message' => $e->getMessage()], 500);
        }
    }



    public function getMacros(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }



        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch all macro plans associated with the client and coach
        $macroPlans = macroPlans::where('client_assigned_id', $request->client_id)
                               ->where('coach_id', $coach_id)
                               ->get();

        return response()->json($macroPlans, 200);
    }


    public function deleteMacro(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
            'macro_plan_id' => 'required|string|max:255|exists:macro_plans,macro_plan_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch the Macro Plan by macro_plan_id
        $macroPlan = macroPlans::where('macro_plan_id', $request->macro_plan_id)
            ->where('client_assigned_id', $request->client_id)
            ->where('coach_id', $coach_id)
            ->first();

        // Ensure the macro plan exists and is associated with the client and coach
        if (!$macroPlan) {
            return response()->json(['error' => 'Macro plan not found or does not belong to the authenticated coach'], 404);
        }

        DB::beginTransaction();

        try {
            // Delete the Macro Plan
            $macroPlan->delete();

            // Create a notification for the macro deletion
            NotificationHelper::createNotification($request->client_id, $coach_id, 2, $macroPlan->macro_plan_id); // 2: "Macro Plan Updated"

            DB::commit();

            return response()->json(['success' => 'Macro plan deleted successfully'], 200);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Failed to delete macro plan', 'message' => $e->getMessage()], 500);
        }
    }





    public function deleteExercise(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
            'workout_plan_id' => 'required|string|max:255|exists:workout_plans,workout_plan_id',
            'workout_day_id' => 'required|string|max:255|exists:workout_days,workout_day_id',
            'exercise_id' => 'required|string|max:255|exists:exercises,exercise_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch the Workout Plan and verify it belongs to the coach and client
        $workoutPlan = workoutPlans::where('workout_plan_id', $request->workout_plan_id)
                                  ->where('coach_id', $coach_id)
                                  ->where('client_assigned_id', $request->client_id)
                                  ->first();

        if (!$workoutPlan) {
            return response()->json(['error' => 'Workout plan not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch the Workout Day and verify it belongs to the workout plan
        $workoutDay = workoutDays::where('workout_day_id', $request->workout_day_id)
                                ->where('workout_plan_id', $request->workout_plan_id)
                                ->first();

        if (!$workoutDay) {
            return response()->json(['error' => 'Workout day not found or does not belong to the specified workout plan'], 404);
        }

        // Fetch the Exercise and verify it belongs to the workout day
        $exercise = exercises::where('exercise_id', $request->exercise_id)
                            ->where('workout_day_id', $request->workout_day_id)
                            ->first();

        if (!$exercise) {
            return response()->json(['error' => 'Exercise not found or does not belong to the specified workout day'], 404);
        }

        // Delete the associated rep sets
        $exercise->repsSets()->delete();

        // Delete the Exercise
        $exercise->delete();

        // Create a notification for the exercise deletion
        NotificationHelper::createNotification($request->client_id, $coach_id, 3, $workoutDay->workout_plan_id); // 3: "Exercise Deleted"

        return response()->json(['success' => 'Exercise and associated rep sets deleted successfully'], 200);
    }



    public function deleteDay(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
            'workout_plan_id' => 'required|string|max:255|exists:workout_plans,workout_plan_id',
            'workout_day_id' => 'required|string|max:255|exists:workout_days,workout_day_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch the Workout Plan and verify it belongs to the coach and client
        $workoutPlan = workoutPlans::where('workout_plan_id', $request->workout_plan_id)
                                  ->where('coach_id', $coach_id)
                                  ->where('client_assigned_id', $request->client_id)
                                  ->first();

        if (!$workoutPlan) {
            return response()->json(['error' => 'Workout plan not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch the Workout Day and verify it belongs to the workout plan
        $workoutDay = workoutDays::where('workout_day_id', $request->workout_day_id)
                                ->where('workout_plan_id', $request->workout_plan_id)
                                ->first();

        if (!$workoutDay) {
            return response()->json(['error' => 'Workout day not found or does not belong to the specified workout plan'], 404);
        }

        // Create a notification for the workout day deletion
        NotificationHelper::createNotification($request->client_id, $coach_id, 4, $workoutDay->workout_plan_id); // 4: "Workout Day Deleted"

        // Delete all associated exercises and their rep sets
        foreach ($workoutDay->exercises as $exercise) {
            $exercise->repsSets()->delete();  // Delete associated rep sets
            $exercise->delete();  // Delete the exercise
        }

        // Delete the Workout Day
        $workoutDay->delete();

        return response()->json(['success' => 'Workout day and all associated exercises and rep sets deleted successfully'], 200);
    }


    public function getWorkout(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch workout plans associated with the client and coach
        $workoutPlans = workoutPlans::where('coach_id', $coach_id)
                                    ->where('client_assigned_id', $request->client_id)
                                    ->with(['workoutDays.exercises.repsSets'])
                                    ->get();

        if ($workoutPlans->isEmpty()) {
            return response()->json([], 200);
        }

        // Format the response data to include all related tables' data
        $response = [];
        foreach ($workoutPlans as $plan) {
            $planDetails = [
                'workout_plan_id' => $plan->workout_plan_id,
                'coach_id' => $plan->coach_id,
                'client_assigned_id' => $plan->client_assigned_id,
                'created_at' => $plan->created_at,
                'updated_at' => $plan->updated_at,
                'workout_days' => []
            ];

            foreach ($plan->workoutDays as $day) {
                $dayDetails = [
                    'workout_day_id' => $day->workout_day_id,
                    'day' => $day->day,
                    'created_at' => $day->created_at,
                    'updated_at' => $day->updated_at,
                    'exercises' => []
                ];
                
                foreach ($day->exercises as $exercise) {
                    $exerciseDetails = [
                        'exercise_id' => $exercise->exercise_id,
                        'exercise_name' => $exercise->exercise_name,
                        'sets' => $exercise->sets,
                        'workout_video' => $exercise->workout_video,
                        'note' => $exercise->note,
                        'created_at' => $exercise->created_at,
                        'updated_at' => $exercise->updated_at,
                        'reps_sets' => []
                    ];
                    
                    foreach ($exercise->repsSets as $repsSet) {
                        $exerciseDetails['reps_sets'][] = [
                            'reps_set_id' => $repsSet->reps_set_id,
                            'set' => $repsSet->set,
                            'reps' => $repsSet->reps,
                            'created_at' => $repsSet->created_at,
                            'updated_at' => $repsSet->updated_at
                        ];
                    }
                    
                    $dayDetails['exercises'][] = $exerciseDetails;
                }
                
                $planDetails['workout_days'][] = $dayDetails;
            }

            $response[] = $planDetails;
        }

        return response()->json(['success' => 'Workout plans retrieved successfully', 'details' => $response], 200);
    }



    public function assignWorkout(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
            'details' => 'required|string' // Validate details as a string
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $detailsString = $request->details;

        // Decode the JSON string to an array
        $details = json_decode($detailsString, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return response()->json(['error' => 'Invalid JSON format in details'], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        DB::beginTransaction();

        try {
            // Fetch existing workout plans for the client
            $existingPlans = workoutPlans::where('coach_id', $coach_id)
                                        ->where('client_assigned_id', $request->client_id)
                                        ->with(['workoutDays.exercises.repsSets'])
                                        ->get();

            // Flag to determine if new workouts are created
            $newWorkoutsCreated = true;

            // Delete related data if any existing plans
            if ($existingPlans->isNotEmpty()) {
                $newWorkoutsCreated = false; // Setting to false since we are updating

                foreach ($existingPlans as $plan) {
                    foreach ($plan->workoutDays as $day) {
                        foreach ($day->exercises as $exercise) {
                            // Delete repsSets first
                            $exercise->repsSets()->delete();
                        }
                        // Delete exercises
                        $day->exercises()->delete();
                        // Delete workoutDays
                        $day->delete();
                    }
                    // Delete workoutPlans
                    $plan->delete();
                }
            }

            // Insert new workout plans
            if ($details) {
                $newWorkoutPlan = workoutPlans::create([
                    'coach_id' => $coach_id,
                    'client_assigned_id' => $request->client_id,
                ]);

                foreach ($details as $dayDetail) {
                    $workoutDay = workoutDays::create([
                        'workout_plan_id' => $newWorkoutPlan->workout_plan_id,
                        'day' => $dayDetail['day']
                    ]);

                    foreach ($dayDetail['exercises'] as $exerciseDetail) {
                        $exercise = exercises::create([
                            'workout_day_id' => $workoutDay->workout_day_id,
                            'exercise_name' => $exerciseDetail['exercise_name'],
                            'sets' => $exerciseDetail['sets'],
                            'workout_video' => $exerciseDetail['workout_video'],
                            'note' => $exerciseDetail['note']
                        ]);

                        foreach ($exerciseDetail['reps_sets'] as $repsSet) {
                            repsSets::create([
                                'exercise_id' => $exercise->exercise_id,
                                'set' => $repsSet['set'],
                                'reps' => $repsSet['reps']
                            ]);
                        }
                    }
                }
            }

            // Update the client's 'is_training_assigned' flag to true
            $client->update(['is_training_assigned' => true]);

            // Create a notification based on whether it's a new workout assignment or an update
            $notificationType = $newWorkoutsCreated ? 3 : 4; // 3: "New Workout Assigned", 4: "Workout Plan Updated"
            NotificationHelper::createNotification($request->client_id, $coach_id, $notificationType, $newWorkoutPlan->workout_plan_id);

            DB::commit();

            return response()->json(['success' => 'Workout plan assigned successfully'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'An error occurred while assigning the workout plan'], 500);
        }
    }



    public function getClientDetails(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coachId = $session->user_id;

        // // Fetch client details based on coach_id and client_id
        // $clients = Clients::where('coach_id', $coachId)
        //     ->with([
        //         'checkInDetails.checkInPictures',
        //         'macroPlans' => function ($query) use ($coachId) {
        //             $query->where('coach_id', $coachId);
        //         },
        //         'workoutPlans.workoutDays.exercises.repsSets' => function ($query) use ($coachId) {
        //             $query->whereHas('workoutPlan', function ($query) use ($coachId) {
        //                 $query->where('coach_id', $coachId);
        //             });
        //         }
        //     ])
        //     ->get();

        $clients = Clients::where('coach_id', $coachId)
            ->with([
                'checkInDetails.checkInPictures',
                'macroPlans' => function ($query) use ($coachId) {
                    $query->where('coach_id', $coachId);
                },
                'workoutPlans.workoutDays.exercises.repsSets.exercise.workoutDay.workoutPlan' => function ($query) use ($coachId) {
                    $query->where('coach_id', $coachId);
                }
            ])
            ->get();



        if ($clients->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No clients found for this coach'
            ], 404);
        }

        // Return the response as a JSON array
        return response()->json([
            'status' => 'success',
            'clients' => $clients
        ], 200);
    }



    public function getCheckInPictures(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Validate the request parameters
        $validator = Validator::make($request->all(), [
            'check_in_details_id' => 'required|integer|exists:check_in_details,details_id', 
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first()
            ], 400);
        }

        // Fetch the validated check_in_details_id
        // $checkInDetailsId = $request->input('check_in_details_id');
        $checkInDetailsId = $request->check_in_details_id;

        // Fetch all pictures for the specified check_in_details_id
        $pictures = CheckInPictures::where('check_in_details_id', $checkInDetailsId)->get();

        if ($pictures->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No pictures found for this check-in details ID'
            ], 404);
        }

        // Return the pictures as a JSON array
        return response()->json([
            'status' => 'success',
            'pictures' => $pictures
        ], 200);
    }




    public function addFeedback(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Validate the request parameters
        $validator = Validator::make($request->all(), [
            'details_id' => 'required|integer|exists:check_in_details,details_id',
            'feedback' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first()
            ], 400);
        }

        // Fetch the validated data
        $detailsId = $request->details_id;
        $feedback = $request->feedback;

        // Find the CheckInDetails record
        $checkInDetails = checkInDetails::find($detailsId);

        if (!$checkInDetails) {
            return response()->json([
                'status' => 'error',
                'message' => 'Check-in details not found'
            ], 404);
        }

        // Add the feedback to the check-in details (assuming a coach_feedback field exists)
        $checkInDetails->coach_feedback = $feedback;
        $checkInDetails->save();

        return response()->json([
            'status' => 'success',
            'message' => 'Feedback added successfully'
        ], 200);
    }









    // public function assignDietPlan(Request $request)
    // {
    //     // Extract the token from the Authorization header
    //     $token = $request->bearerToken();

    //     if (!$token) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Token not provided'
    //         ], 401);
    //     }

    //     // Find the session based on the provided authentication token
    //     $session = Session::where('authentication_token', $token)->first();

    //     if (!$session) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token or session not found'
    //         ], 401);
    //     }

    //     // Retrieve the coach_id from the session
    //     $coach_id = $session->user_id;

    //     // Validate the incoming request
    //     $validator = Validator::make($request->all(), [
    //         'client_id' => 'required|integer|exists:clients,client_id',
    //         'details' => 'required|string'  // Expect details as a JSON string
    //     ]);

    //     if ($validator->fails()) {
    //         return response()->json(['error' => $validator->errors()->first()], 400);
    //     }

    //     $detailsString = $request->details;

    //     // Decode the JSON string to an array
    //     $details = json_decode($detailsString, true);

    //     if (json_last_error() !== JSON_ERROR_NONE) {
    //         return response()->json(['error' => 'Invalid JSON format in details'], 400);
    //     }

    //     // Check if the client belongs to the authenticated coach
    //     $client = Clients::where('client_id', $request->client_id)
    //         ->where('coach_id', $coach_id)
    //         ->first();

    //     if (!$client) {
    //         return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
    //     }

    //     DB::beginTransaction();

    //     try {
    //         // Fetch existing diet plans for the client
    //         $existingPlans = dietPlans::where('coach_id', $coach_id)
    //             ->where('assigned_to', $request->client_id)
    //             ->with(['dietDays.meals'])
    //             ->get();

    //         // Determine if new diet plans are created
    //         $newDietPlansCreated = $existingPlans->isEmpty();

    //         // If there are existing plans, delete them
    //         if ($existingPlans->isNotEmpty()) {
    //             foreach ($existingPlans as $plan) {
    //                 foreach ($plan->dietDays as $day) {
    //                     foreach ($day->meals as $meal) {
    //                         $meal->delete();
    //                     }
    //                     $day->delete();
    //                 }
    //                 $plan->delete();
    //             }
    //         }

    //         // Insert new diet plan
    //         $newDietPlan = dietPlans::create([
    //             'coach_id' => $coach_id,
    //             'assigned_to' => $request->client_id,
    //         ]);

    //         foreach ($details as $dayDetail) {
    //             // Create a new diet day
    //             $dietDay = dietDays::create([
    //                 'diet_plan_id' => $newDietPlan->diet_plan_id,
    //                 'day' => $dayDetail['day']
    //             ]);

    //             foreach ($dayDetail['meals'] as $mealDetail) {
    //                 // Insert meals for the diet day
    //                 Meals::create([
    //                     'diet_day_id' => $dietDay->diet_day_id,
    //                     'meal_title' => $mealDetail['meal_title'],
    //                     'serving_size' => $mealDetail['serving_size'],
    //                     'serving_unit' => $mealDetail['serving_unit'],
    //                     'fat' => $mealDetail['fat'],
    //                     'carbohydrates' => $mealDetail['carbohydrates'],
    //                     'protein' => $mealDetail['protein'],
    //                     'fibre' => $mealDetail['fibre'],
    //                     'calories' => $mealDetail['calories'],
    //                     'sugar' => $mealDetail['sugar'],
    //                     'sodium' => $mealDetail['sodium'],
    //                     'note' => $mealDetail['note']
    //                 ]);
    //             }
    //         }

    //         // Create a notification based on whether it's a new diet assignment or an update
    //         $notificationType = $newDietPlansCreated ? 5 : 6; // 5: "New Diet Assigned", 6: "Diet Plan Updated"
    //         NotificationHelper::createNotification($request->client_id, $coach_id, $notificationType, $newDietPlan->diet_plan_id);

    //         // Update the client's 'is_diet_assigned' flag to true
    //         $client->update(['is_diet_assigned' => true]);

    //         DB::commit();

    //         return response()->json(['success' => 'Diet plan assigned successfully'], 200);

    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         return response()->json(['error' => 'An error occurred while assigning the diet plan'], 500);
    //     }
    // }

    public function assignDietPlan(Request $request) 
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|integer|exists:clients,client_id',
            'details' => 'required|string'  // Expect details as a JSON string
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $detailsString = $request->details;

        // Decode the JSON string to an array
        $details = json_decode($detailsString, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return response()->json(['error' => 'Invalid JSON format in details'], 400);
        }

        // Check if the client belongs to the authenticated coach
        $client = Clients::where('client_id', $request->client_id)
            ->where('coach_id', $coach_id)
            ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        DB::beginTransaction();

        try {
            // Fetch existing diet plans for the client
            $existingPlans = dietPlans::where('coach_id', $coach_id)
                ->where('assigned_to', $request->client_id)
                ->with(['dietDays.meals'])
                ->get();

            // Determine if new diet plans are created
            $newDietPlansCreated = $existingPlans->isEmpty();

            // If there are existing plans, delete them
            if ($existingPlans->isNotEmpty()) {
                foreach ($existingPlans as $plan) {
                    foreach ($plan->dietDays as $day) {
                        foreach ($day->meals as $meal) {
                            $meal->mealItems()->delete(); // Delete associated meal items
                            $meal->delete();
                        }
                        $day->delete();
                    }
                    $plan->delete();
                }
            }

            // Insert new diet plan
            $newDietPlan = dietPlans::create([
                'coach_id' => $coach_id,
                'assigned_to' => $request->client_id,
            ]);


            foreach ($details as $dayDetail) {
                // Create a new diet day
                $dietDay = dietDays::create([
                    'diet_plan_id' => $newDietPlan->diet_plan_id,
                    'day' => $dayDetail['day']
                ]);

                foreach ($dayDetail['meals'] as $mealDetail) {
                    // Insert meal
                    $meal = Meals::create([
                        'diet_day_id' => $dietDay->diet_day_id,
                        'meal_title' => $mealDetail['meal_title'],
                    ]);

                    // Insert multiple meal items for the meal
                    foreach ($mealDetail['items'] as $itemDetail) {
                        MealItems::create([
                            'meal_id' => $meal->meal_id,
                            'item_title' => $itemDetail['item_title'],
                            'serving_size' => $itemDetail['serving_size'],
                            'serving_unit' => $itemDetail['serving_unit'],
                            'fat' => $itemDetail['fat'],
                            'carbohydrates' => $itemDetail['carbohydrates'],
                            'protein' => $itemDetail['protein'],
                            'fibre' => $itemDetail['fibre'],
                            'calories' => $itemDetail['calories'],
                            'sugar' => $itemDetail['sugar'],
                            'sodium' => $itemDetail['sodium'],
                            'note' => $itemDetail['note'],
                        ]);
                    }
                }
            }

            // Create a notification based on whether it's a new diet assignment or an update
            $notificationType = $newDietPlansCreated ? 5 : 6; // 5: "New Diet Assigned", 6: "Diet Plan Updated"
            NotificationHelper::createNotification($request->client_id, $coach_id, $notificationType, $newDietPlan->diet_plan_id);

            // Update the client's 'is_diet_assigned' flag to true
            $client->update(['is_diet_assigned' => true]);

            DB::commit();

            return response()->json(['success' => 'Diet plan assigned successfully'], 200);

        } catch (\Exception $e) {
            \Log::error("Error assigning diet plan: " . $e->getMessage());
            DB::rollBack();
            return response()->json(['error' => 'An error occurred while assigning the diet plan', $e->getMessage()], 500);
        }
    }



    // public function getDietPlan(Request $request)
    // {
    //     // Extract the token from the Authorization header
    //     $token = $request->bearerToken();

    //     if (!$token) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Token not provided'
    //         ], 401);
    //     }

    //     // Find the session based on the provided authentication token
    //     $session = Session::where('authentication_token', $token)->first();

    //     if (!$session) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token or session not found'
    //         ], 401);
    //     }

    //     // Retrieve the coach_id from the session
    //     $coach_id = $session->user_id;

    //     // Validate the incoming request
    //     $validator = Validator::make($request->all(), [
    //         'client_id' => 'required|string|max:255|exists:clients,client_id',
    //     ]);

    //     // Return validation errors
    //     if ($validator->fails()) {
    //         return response()->json(['error' => $validator->errors()->first()], 400);
    //     }

    //     // Fetch the client and verify they belong to the coach
    //     $client = Clients::where('client_id', $request->client_id)
    //                     ->where('coach_id', $coach_id)
    //                     ->first();

    //     if (!$client) {
    //         return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
    //     }

    //     // Fetch diet plans associated with the client and coach
    //     $dietPlans = dietPlans::where('coach_id', $coach_id)
    //                           ->where('assigned_to', $request->client_id)
    //                           ->with(['dietDays.meals'])
    //                           ->get();

    //     if ($dietPlans->isEmpty()) {
    //         return response()->json([], 200);
    //     }

    //     // Format the response data to include all related tables' data
    //     $response = [];
    //     foreach ($dietPlans as $plan) {
    //         $planDetails = [
    //             'diet_plan_id' => $plan->diet_plan_id,
    //             'coach_id' => $plan->coach_id,
    //             'assigned_to' => $plan->assigned_to,
    //             'created_at' => $plan->created_at,
    //             'updated_at' => $plan->updated_at,
    //             'diet_days' => []
    //         ];

    //         foreach ($plan->dietDays as $day) {
    //             $dayDetails = [
    //                 'diet_day_id' => $day->diet_day_id,
    //                 'day' => $day->day,
    //                 'created_at' => $day->created_at,
    //                 'updated_at' => $day->updated_at,
    //                 'meals' => []
    //             ];
                
    //             foreach ($day->meals as $meal) {
    //                 $mealDetails = [
    //                     'meal_id' => $meal->meal_id,
    //                     'meal_title' => $meal->meal_title,
    //                     'serving_size' => $meal->serving_size,
    //                     'serving_unit' => $meal->serving_unit,
    //                     'fat' => $meal->fat,
    //                     'carbohydrates' => $meal->carbohydrates,
    //                     'protein' => $meal->protein,
    //                     'fibre' => $meal->fibre,
    //                     'calories' => $meal->calories,
    //                     'sugar' => $meal->sugar,
    //                     'sodium' => $meal->sodium,
    //                     'note' => $meal->note,
    //                     'created_at' => $meal->created_at,
    //                     'updated_at' => $meal->updated_at,
    //                 ];

    //                 $dayDetails['meals'][] = $mealDetails;
    //             }

    //             $planDetails['diet_days'][] = $dayDetails;
    //         }

    //         $response[] = $planDetails;
    //     }

    //     return response()->json(['success' => 'Diet plans retrieved successfully', 'details' => $response], 200);
    // }

    public function getDietPlan(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|string|max:255|exists:clients,client_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Fetch diet plans associated with the client and coach
        $dietPlans = dietPlans::where('coach_id', $coach_id)
                              ->where('assigned_to', $request->client_id)
                              ->with('dietDays.meals.mealItems') // Load mealItems for each meal
                              ->get();


        if ($dietPlans->isEmpty()) {
            return response()->json([], 200);
        }

        // Format the response data to include all related tables' data
        $response = [];
        foreach ($dietPlans as $plan) {
            $planDetails = [
                'diet_plan_id' => $plan->diet_plan_id,
                'coach_id' => $plan->coach_id,
                'assigned_to' => $plan->assigned_to,
                'created_at' => $plan->created_at,
                'updated_at' => $plan->updated_at,
                'diet_days' => []
            ];

            foreach ($plan->dietDays as $day) {
                $dayDetails = [
                    'diet_day_id' => $day->diet_day_id,
                    'day' => $day->day,
                    'created_at' => $day->created_at,
                    'updated_at' => $day->updated_at,
                    'meals' => []
                ];

                foreach ($day->meals as $meal) {
                    $mealDetails = [
                        'meal_id' => $meal->meal_id,
                        'meal_title' => $meal->meal_title,
                        'created_at' => $meal->created_at,
                        'updated_at' => $meal->updated_at,
                        'meal_items' => []
                    ];

                    // Loop through meal items to get their details
                    foreach ($meal->mealItems as $mealItem) {
                        $mealDetails['meal_items'][] = [
                            'item_id' => $mealItem->meal_items_id,
                            'item_title' => $mealItem->item_title,
                            'serving_size' => $mealItem->serving_size,
                            'serving_unit' => $mealItem->serving_unit,
                            'fat' => $mealItem->fat,
                            'carbohydrates' => $mealItem->carbohydrates,
                            'protein' => $mealItem->protein,
                            'fibre' => $mealItem->fibre,
                            'calories' => $mealItem->calories,
                            'sugar' => $mealItem->sugar,
                            'sodium' => $mealItem->sodium,
                            'note' => $mealItem->note,
                            'created_at' => $mealItem->created_at,
                            'updated_at' => $mealItem->updated_at,
                        ];
                    }

                    $dayDetails['meals'][] = $mealDetails;
                }

                $planDetails['diet_days'][] = $dayDetails;
            }


            $response[] = $planDetails;
        }

        return response()->json(['success' => 'Diet plans retrieved successfully', 'details' => $response], 200);
    }


    // public function deleteMeal(Request $request)
    // {
    //     // Extract the token from the Authorization header
    //     $token = $request->bearerToken();

    //     if (!$token) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Token not provided'
    //         ], 401);
    //     }

    //     // Find the session based on the provided authentication token
    //     $session = Session::where('authentication_token', $token)->first();

    //     if (!$session) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token or session not found'
    //         ], 401);
    //     }

    //     // Retrieve the coach_id from the session
    //     $coach_id = $session->user_id;

    //     // Validate the incoming request
    //     $validator = Validator::make($request->all(), [
    //         'client_id' => 'required|integer|exists:clients,client_id',
    //         'diet_plan_id' => 'required|integer|exists:diet_plans,diet_plan_id',
    //         'diet_day_id' => 'required|integer|exists:diet_days,diet_day_id',
    //         'meal_id' => 'required|integer|exists:meals,meal_id',
    //     ]);

    //     // Return validation errors
    //     if ($validator->fails()) {
    //         return response()->json(['error' => $validator->errors()->first()], 400);
    //     }

    //     // Fetch the client and verify they belong to the coach
    //     $client = Clients::where('client_id', $request->client_id)
    //                     ->where('coach_id', $coach_id)
    //                     ->first();

    //     if (!$client) {
    //         return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
    //     }

    //     // Verify that the diet plan, diet day, and meal belong to the correct client and coach
    //     $dietPlan = dietPlans::where('diet_plan_id', $request->diet_plan_id)
    //                          ->where('coach_id', $coach_id)
    //                          ->where('assigned_to', $request->client_id)
    //                          ->first();

    //     if (!$dietPlan) {
    //         return response()->json(['error' => 'Diet plan not found or does not belong to the authenticated coach/client'], 404);
    //     }

    //     $dietDay = dietDays::where('diet_day_id', $request->diet_day_id)
    //                        ->where('diet_plan_id', $dietPlan->diet_plan_id)
    //                        ->first();

    //     if (!$dietDay) {
    //         return response()->json(['error' => 'Diet day not found or does not belong to the diet plan'], 404);
    //     }

    //     // Find the meal and delete it
    //     $meal = Meals::where('meal_id', $request->meal_id)
    //                  ->where('diet_day_id', $dietDay->diet_day_id)
    //                  ->first();

    //     if (!$meal) {
    //         return response()->json(['error' => 'Meal not found or does not belong to the diet day'], 404);
    //     }

    //     try {
    //         $meal->delete();

    //         // Create a notification for the deletion
    //         NotificationHelper::createNotification($request->client_id, $coach_id, 6, $dietPlan->diet_plan_id);

    //         return response()->json(['success' => 'Meal deleted successfully'], 200);

    //     } catch (\Exception $e) {
    //         return response()->json(['error' => 'An error occurred while deleting the meal'], 500);
    //     }
    // }


    public function deleteMeal(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|integer|exists:clients,client_id',
            'diet_plan_id' => 'required|integer|exists:diet_plans,diet_plan_id',
            'diet_day_id' => 'required|integer|exists:diet_days,diet_day_id',
            'meal_id' => 'required|integer|exists:meals,meal_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Verify that the diet plan, diet day, and meal belong to the correct client and coach
        $dietPlan = dietPlans::where('diet_plan_id', $request->diet_plan_id)
                             ->where('coach_id', $coach_id)
                             ->where('assigned_to', $request->client_id)
                             ->first();

        if (!$dietPlan) {
            return response()->json(['error' => 'Diet plan not found or does not belong to the authenticated coach/client'], 404);
        }

        $dietDay = dietDays::where('diet_day_id', $request->diet_day_id)
                           ->where('diet_plan_id', $dietPlan->diet_plan_id)
                           ->first();

        if (!$dietDay) {
            return response()->json(['error' => 'Diet day not found or does not belong to the diet plan'], 404);
        }

        // Check if meal_items exist, then delete them if necessary
        $mealItems = MealItems::where('meal_id', $request->meal_id)->get();

        if ($mealItems->isNotEmpty()) {
            // Delete associated meal items first
            foreach ($mealItems as $mealItem) {
                $mealItem->delete();
            }
        }

        // Now delete the meal
        $meal = Meals::where('meal_id', $request->meal_id)
                     ->where('diet_day_id', $dietDay->diet_day_id)
                     ->first();

        if (!$meal) {
            return response()->json(['error' => 'Meal not found or does not belong to the diet day'], 404);
        }

        try {
            $meal->delete();

            // Create a notification for the deletion
            NotificationHelper::createNotification($request->client_id, $coach_id, 6, $dietPlan->diet_plan_id);

                return response()->json(['success' => 'Meal and related items deleted successfully'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while deleting the meal or meal items'], 500);
        }
    }



    public function deleteMealItem(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|integer|exists:clients,client_id',
            'diet_plan_id' => 'required|integer|exists:diet_plans,diet_plan_id',
            'diet_day_id' => 'required|integer|exists:diet_days,diet_day_id',
            'meal_id' => 'required|integer|exists:meals,meal_id',
            'meal_items_id' => 'required|integer|exists:meal_items,meal_items_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                         ->where('coach_id', $coach_id)
                         ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Verify that the diet plan, diet day, and meal belong to the correct client and coach
        $dietPlan = dietPlans::where('diet_plan_id', $request->diet_plan_id)
                             ->where('coach_id', $coach_id)
                             ->where('assigned_to', $request->client_id)
                             ->first();

        if (!$dietPlan) {
            return response()->json(['error' => 'Diet plan not found or does not belong to the authenticated coach/client'], 404);
        }

        $dietDay = dietDays::where('diet_day_id', $request->diet_day_id)
                           ->where('diet_plan_id', $dietPlan->diet_plan_id)
                           ->first();

        if (!$dietDay) {
            return response()->json(['error' => 'Diet day not found or does not belong to the diet plan'], 404);
        }

        // Verify the meal and delete the meal item
        $mealItem = MealItems::where('meal_items_id', $request->meal_items_id)
                             ->where('meal_id', $request->meal_id)
                             ->first();

        if (!$mealItem) {
            return response()->json(['error' => 'Meal item not found'], 404);
        }

        try {
            // Delete the meal item
            $mealItem->delete();

            // Optionally, you can notify about the deletion
            NotificationHelper::createNotification($request->client_id, $coach_id, 7, $dietPlan->diet_plan_id);

            return response()->json(['success' => 'Meal item deleted successfully'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while deleting the meal item'], 500);
        }
    }




    // public function deleteDietDay(Request $request)
    // {
    //     // Extract the token from the Authorization header
    //     $token = $request->bearerToken();

    //     if (!$token) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Token not provided'
    //         ], 401);
    //     }

    //     // Find the session based on the provided authentication token
    //     $session = Session::where('authentication_token', $token)->first();

    //     if (!$session) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token or session not found'
    //         ], 401);
    //     }

    //     // Retrieve the coach_id from the session
    //     $coach_id = $session->user_id;

    //     // Validate the incoming request
    //     $validator = Validator::make($request->all(), [
    //         'client_id' => 'required|integer|exists:clients,client_id',
    //         'diet_plan_id' => 'required|integer|exists:diet_plans,diet_plan_id',
    //         'diet_day_id' => 'required|integer|exists:diet_days,diet_day_id',
    //     ]);

    //     // Return validation errors
    //     if ($validator->fails()) {
    //         return response()->json(['error' => $validator->errors()->first()], 400);
    //     }

    //     // Fetch the client and verify they belong to the coach
    //     $client = Clients::where('client_id', $request->client_id)
    //                     ->where('coach_id', $coach_id)
    //                     ->first();

    //     if (!$client) {
    //         return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
    //     }

    //     // Verify that the diet plan belongs to the correct client and coach
    //     $dietPlan = dietPlans::where('diet_plan_id', $request->diet_plan_id)
    //                          ->where('coach_id', $coach_id)
    //                          ->where('assigned_to', $request->client_id)
    //                          ->first();

    //     if (!$dietPlan) {
    //         return response()->json(['error' => 'Diet plan not found or does not belong to the authenticated coach/client'], 404);
    //     }

    //     // Find the diet day and ensure it belongs to the specified diet plan
    //     $dietDay = dietDays::where('diet_day_id', $request->diet_day_id)
    //                        ->where('diet_plan_id', $dietPlan->diet_plan_id)
    //                        ->first();

    //     if (!$dietDay) {
    //         return response()->json(['error' => 'Diet day not found or does not belong to the diet plan'], 404);
    //     }

    //     DB::beginTransaction();

    //     try {
    //         // Delete associated meals first
    //         $dietDay->meals()->delete();

    //         // Then delete the diet day
    //         $dietDay->delete();

    //         // Create a notification for the deletion
    //         NotificationHelper::createNotification($request->client_id, $coach_id, 6, $dietPlan->diet_plan_id);

    //         DB::commit();

    //         return response()->json(['success' => 'Diet day and associated meals deleted successfully'], 200);

    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         return response()->json(['error' => 'An error occurred while deleting the diet day'], 500);
    //     }
    // }

    public function deleteDietDay(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|integer|exists:clients,client_id',
            'diet_plan_id' => 'required|integer|exists:diet_plans,diet_plan_id',
            'diet_day_id' => 'required|integer|exists:diet_days,diet_day_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client and verify they belong to the coach
        $client = Clients::where('client_id', $request->client_id)
                        ->where('coach_id', $coach_id)
                        ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to the authenticated coach'], 404);
        }

        // Verify that the diet plan belongs to the correct client and coach
        $dietPlan = dietPlans::where('diet_plan_id', $request->diet_plan_id)
                             ->where('coach_id', $coach_id)
                             ->where('assigned_to', $request->client_id)
                             ->first();

        if (!$dietPlan) {
            return response()->json(['error' => 'Diet plan not found or does not belong to the authenticated coach/client'], 404);
        }

        // Find the diet day and ensure it belongs to the specified diet plan
        $dietDay = dietDays::where('diet_day_id', $request->diet_day_id)
                           ->where('diet_plan_id', $dietPlan->diet_plan_id)
                           ->first();

        if (!$dietDay) {
            return response()->json(['error' => 'Diet day not found or does not belong to the diet plan'], 404);
        }

        DB::beginTransaction();

        try {
            // Delete associated meal items first
            $dietDay->meals->each(function ($meal) {
                $meal->mealItems()->delete();
            });

            // Then delete associated meals
            $dietDay->meals()->delete();

            // Then delete the diet day
            $dietDay->delete();

            // Create a notification for the deletion
            NotificationHelper::createNotification($request->client_id, $coach_id, 6, $dietPlan->diet_plan_id);

            DB::commit();

            return response()->json(['success' => 'Diet day, meals, and associated meal items deleted successfully'], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'An error occurred while deleting the diet day'], 500);
        }
    }



    public function addClientPreference(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the client_id from the session
        $client_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|integer|exists:coaches,coach_id', // Ensure coach_id exists in coaches table
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Begin a database transaction
        DB::beginTransaction();

        try {
            // Create a new client preference record
            clientPreferences::create([
                'client_id' => $client_id,
                'coach_id' => $request->coach_id,
                // 'coach_status' => null, // Initially, this can be null
                // 'client_status' => null, // Initially, this can be null
                'message' => 'looking for a coach', // Set the message
            ]);

            DB::commit();

            return response()->json(['success' => 'Client preference added successfully'], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            // Return the specific error message for debugging
            return response()->json([
                'error' => 'An error occurred while adding the client preference',
                'exception' => $e->getMessage() // Add the exception message
            ], 500);

        }
    }


    public function getClientPreferencesForCoach(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Set default pagination values
        $page = $request->input('page', 1); // default to 1 if not provided
        $limit = $request->input('limit', 10); // default to 10 if not provided

        // Calculate the offset for the query
        $offset = ($page - 1) * $limit;

        // Fetch client preferences where the coach_id matches with pagination
        $clientPreferencesQuery = clientPreferences::where('coach_id', $coach_id)
                                                  ->with('client'); // Include client details

        // Get total count of client preferences
        $totalPreferences = $clientPreferencesQuery->count();

        // Get paginated client preferences
        $clientPreferences = $clientPreferencesQuery->skip($offset)->take($limit)->get();

        if ($clientPreferences->isEmpty()) {
            return response()->json(['message' => 'No client preferences found for this coach'], 404);
        }

        // Format the response data
        $response = [];
        foreach ($clientPreferences as $preference) {
            $response[] = [
                'client_preferences_id' => $preference->client_preferences_id,
                'client_id' => $preference->client_id,
                'coach_id' => $preference->coach_id,
                'coach_status' => $preference->coach_status,
                'client_status' => $preference->client_status,
                'message' => $preference->message,
                'client_details' => $preference->client->toArray() // Fetch all client details
            ];
        }

        // Calculate total pages
        $totalPages = ceil($totalPreferences / $limit);

        // Return the response as a JSON object with pagination metadata
        return response()->json([
            'success' => 'Client preferences retrieved successfully',
            'meta' => [
                'page' => $page,
                'pages' => $totalPages,
                'limit' => $limit,
                'total' => $totalPreferences,
            ],
            'details' => $response
        ], 200);
    }

    public function getClientPreferencesForClient(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the client_id from the session
        $client_id = $session->user_id;

        // Fetch client preferences where the client_id matches
        $clientPreferences = clientPreferences::where('client_id', $client_id)
                                              ->with('coach') // Include coach details
                                              ->get();

        if ($clientPreferences->isEmpty()) {
            return response()->json(['status' => 'error', 'message' => 'No client preferences found for this client'], 404);
        }

        // Format the response data
        $response = [];
        foreach ($clientPreferences as $preference) {
            $response[] = [
                'client_preferences_id' => $preference->client_preferences_id,
                'client_id' => $preference->client_id,
                'coach_id' => $preference->coach_id,
                'coach_status' => $preference->coach_status,
                'client_status' => $preference->client_status,
                'message' => $preference->message,
                'coach_details' => $preference->coach->toArray() // Fetch all coach details
            ];
        }

        return response()->json(['status' => 'success', 'message' =>'Client preferences retrieved successfully', 'details' => $response], 200);
    }

    public function updateClientPreferencesForCoach(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|integer|exists:clients,client_id',
            'coach_status' => 'required|string|in:accept,reject',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Fetch the client preference based on client_id and coach_id
        $preference = clientPreferences::where('client_id', $request->client_id)
                                        ->where('coach_id', $coach_id)
                                        ->first();

        if (!$preference) {
            return response()->json(['error' => 'Client preference not found'], 404);
        }

        // Update coach_status and message based on the provided coach_status
        $preference->coach_status = $request->coach_status;
        $preference->message = $request->coach_status === 'accept' ? 'accepted by coach' : 'rejected by coach';
        if ($request->coach_status === 'reject') {
            $preference->active = 0;
        }

        // Save the updated preference
        $preference->save();

        // Create a notification for the update
        $notificationType = $request->coach_status === 'accept' ? 7 : 8; // 7: "Client Accepted", 8: "Client Rejected"
        NotificationHelper::createNotification($request->client_id, $coach_id, $notificationType, $coach_id);

        return response()->json(['success' => 'Client preference updated successfully'], 200);
    }


    public function updateClientPreferencesForClient(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255',
            'client_status' => 'required|in:accept,reject',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Attempt to retrieve the coach by coach_id
        $coach = Coach::where('coach_id', $request->coach_id)->first();

        // Check if the coach exists in the database
        if (!$coach) {
            return $this->errorResponse('No coach found', self::STATUS_NOT_FOUND);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;


            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Check if the client exists
        $client = Clients::find($clientId);
        if (!$client) {
            return response()->json([
                'status' => 'error',
                'message' => 'Client not found',
            ], 404);
        }

        // Begin transaction to ensure atomicity
        DB::beginTransaction();

        try {
            // If the client accepts the coach
            if ($request->client_status === 'accept') {
                // Update the client's coach_id and reset assignments
                $client->coach_id = $request->coach_id;
                $client->is_diet_assigned = false;      // Reset diet assignment
                $client->is_training_assigned = false;  // Reset training assignment
                $client->is_macro_assigned = false;     // Reset macro assignment
                $client->save();

                // Update the messages for all coach preferences
                clientPreferences::where('client_id', $clientId)
                    ->where('coach_id', '!=', $request->coach_id)
                    ->update(['client_status' => 'reject', 'message' => 'selected other coach', 'active' => 0]); // Mark other coaches as rejected and selected other coach

                // Update the message for the accepted coach
                clientPreferences::where('client_id', $clientId)
                    ->where('coach_id', $request->coach_id)
                    ->update(['client_status' => 'accept', 'message' => 'accepted by client', 'active' => 0]); // Mark the selected coach as accepted

            } else {
                // If the client rejects the coach
                clientPreferences::where('client_id', $clientId)
                    ->where('coach_id', $request->coach_id)
                    ->update(['client_status' => 'reject', 'message' => 'rejected by client', 'active' => 0]); // Mark as rejected
            }

            // Commit the transaction
            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Client preferences updated successfully',
            ], 200);
        } catch (\Exception $e) {
            // Rollback the transaction on failure
            DB::rollBack();

            // Log the error for debugging
            \Log::error('Error updating client preferences:', ['error' => $e->getMessage()]);

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update client preferences. Please try again later.',
            ], 500);
        }
    }


    public function getClientDetailsById(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the coach_id from the session
        $coach_id = $session->user_id;

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|integer|exists:clients,client_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Validate the client_id
        $client = Clients::where('client_id', $request->client_id)
                         ->where('coach_id', $coach_id)
                         ->first();

        if (!$client) {
            return response()->json(['error' => 'Client not found or does not belong to this coach'], 404);
        }

        // Attach the coach name
        $coach = Coach::find($client->coach_id);
        $client->coach_name = $coach ? $coach->full_name : null;

        // Fetch check_in_details and associated check_in_pictures
        $check_in_details = checkInDetails::where('client_id', $request->client_id)
                                           ->where('coach_id', $coach_id)
                                           ->with('checkInPictures')
                                           ->orderBy('created_at', 'desc')
                                           ->get();

        // Fetch workout_plans and associated workout_days, exercises, and reps_sets
        $workout_plans = workoutPlans::where('client_assigned_id', $request->client_id)
                                      ->where('coach_id', $coach_id)
                                      ->with(['workoutDays.exercises.repsSets'])
                                      ->orderBy('created_at', 'desc')
                                      ->get();

        // Fetch macro_plans
        $macro_plans = macroPlans::where('client_assigned_id', $request->client_id)
                                 ->where('coach_id', $coach_id)
                                 ->orderBy('created_at', 'desc')
                                 ->get();

        // Fetch diet_plans and associated diet_days, meals, and meal_items
        $diet_plans = dietPlans::where('assigned_to', $request->client_id)
                               ->where('coach_id', $coach_id)
                               ->with(['dietDays.meals.mealItems']) // Include meal items
                               ->orderBy('created_at', 'desc')
                               ->get();

        // Construct response
        $response = [
            'client' => $client,
            'check_in_details' => $check_in_details,
            'workout_plans' => $workout_plans,
            'macro_plans' => $macro_plans,
            'diet_plans' => $diet_plans,
        ];

        return response()->json($response, 200);
    }

    public function getClientDetailsByIdForAdmin(Request $request)
    {
        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'client_id' => 'required|integer|exists:clients,client_id',
            'coach_id' => 'required|integer|exists:coaches,coach_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Validate the client_id
        $client = Clients::where('client_id', $request->client_id)
                         ->where('coach_id', $request->coach_id)
                         ->first();

        if (!$client) {
            return response()->json(['error' => 'Client or coach not found'], 404);
        }

        // Attach the coach name
        $coach = Coach::find($client->coach_id);
        $client->coach_name = $coach ? $coach->full_name : null;

        // Fetch check_in_details and associated check_in_pictures
        $check_in_details = checkInDetails::where('client_id', $request->client_id)
                                           ->where('coach_id', $request->coach_id)
                                           ->with('checkInPictures')
                                           ->orderBy('created_at', 'desc')
                                           ->get();

        // Fetch workout_plans and associated workout_days, exercises, and reps_sets
        $workout_plans = workoutPlans::where('client_assigned_id', $request->client_id)
                                      ->where('coach_id', $request->coach_id)
                                      ->with(['workoutDays.exercises.repsSets'])
                                      ->orderBy('created_at', 'desc')
                                      ->get();

        // Fetch macro_plans
        $macro_plans = macroPlans::where('client_assigned_id', $request->client_id)
                                 ->where('coach_id', $request->coach_id)
                                 ->orderBy('created_at', 'desc')
                                 ->get();

        // Fetch diet_plans and associated diet_days, meals, and meal_items
        $diet_plans = dietPlans::where('assigned_to', $request->client_id)
                               ->where('coach_id', $request->coach_id)
                               ->with(['dietDays.meals.mealItems']) // Include meal items
                               ->orderBy('created_at', 'desc')
                               ->get();

        // Construct response
        $response = [
            'client' => $client,
            'check_in_details' => $check_in_details,
            'workout_plans' => $workout_plans,
            'macro_plans' => $macro_plans,
            'diet_plans' => $diet_plans,
        ];

        return response()->json($response, 200);
    }


    public function searchFood(Request $request)
    {
        // $request->validate([
        //     'search_expression' => 'required|string',
        // ]);

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'search_expression' => 'required|string',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        $searchExpression = $request->input('search_expression');
        $result = $this->foodService->searchFood($searchExpression);

        return response()->json($result);
    }





    public function getAllCoaches(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the user_type from the session
        if ($session->user_type === "admin") {
            // Set default pagination values
            $page = $request->input('page', 1); // default to 1 if not provided
            $limit = $request->input('limit', 10); // default to 10 if not provided

            // Calculate the offset for the query
            $offset = ($page - 1) * $limit;

            // Fetch only coaches with user_role set to 'coach', with pagination
            $coachesQuery = Coach::where('user_role', 'coach')
                ->orderBy('created_at', 'desc');

            // Get total count of coaches
            $totalCoaches = $coachesQuery->count();

            // Get paginated coaches
            $coaches = $coachesQuery->skip($offset)->take($limit)->get();

            // Attach subscription status and append the base URL to intro and profile_picture for each coach
            foreach ($coaches as $coach) {
                // Retrieve subscription status based on the coach_id
                $subscription = Subscription::where('coach_id', $coach->coach_id)->orderBy('created_at', 'desc')->first();
                $coach->subscription_status = $subscription ? $subscription->status : 'inactive';

                // Append base URL to intro if it exists
                if ($coach->intro) {
                    $coach->intro = url($coach->intro);
                }

                // Append base URL to profile_picture if it exists
                if ($coach->profile_picture) {
                    $coach->profile_picture = url($coach->profile_picture);
                }
            }

            // Calculate total pages
            $totalPages = ceil($totalCoaches / $limit);

            // Return the response with pagination metadata
            return response()->json([
                'status' => 'success',
                'meta' => [
                    'page' => $page,
                    'pages' => $totalPages,
                    'limit' => $limit,
                    'total' => $totalCoaches,
                ],
                'coaches' => $coaches,
            ], 200);
        } else {
            return response()->json(['error' => 'Admin not found or does not belong to the authenticated admin'], 404);
        }

        return $this->errorResponse('Failed to fetch all coaches', self::STATUS_INTERNAL_SERVER_ERROR);
    }

    public function getAllClients(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Retrieve the user_type from the session
        if ($session->user_type === "admin") {
            // Set default pagination values
            $page = $request->input('page', 1); // default to 1 if not provided
            $limit = $request->input('limit', 10); // default to 10 if not provided
            $offset = ($page - 1) * $limit;

            // Fetch clients with pagination
            $clientsQuery = Clients::orderBy('created_at', 'desc');
            $totalClients = $clientsQuery->count();
            $clients = $clientsQuery->skip($offset)->take($limit)->get();

            // Attach each client's coach name
            $clients = $clients->map(function ($client) {
                $coach = Coach::find($client->coach_id);
                $client->coach_name = $coach ? $coach->full_name : null;
                return $client;
            });

            // Calculate total pages
            $totalPages = ceil($totalClients / $limit);

            // Return the response with pagination metadata
            return response()->json([
                'status' => 'success',
                'meta' => [
                    'page' => $page,
                    'pages' => $totalPages,
                    'limit' => $limit,
                    'total' => $totalClients,
                ],
                'clients' => $clients,
            ], 200);
        } else {
            return response()->json(['error' => 'Admin not found or does not belong to the authenticated admin'], 404);
        }

        return $this->errorResponse('Failed to fetch all clients', self::STATUS_INTERNAL_SERVER_ERROR);
    }



    public function addFAQ(Request $request)
    {


        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }


        // Retrieve the user_type from the session
        if($session->user_type === "admin"){
            // Validate the incoming request
            $validator = Validator::make($request->all(), [
                'question' => 'required|string',
                'answer' => 'required|string',        
            ]);

            // Return validation errors
            if ($validator->fails()) {
                return response()->json(['error' => $validator->errors()->first()], 400);
            }

            $insertedID = DB::table('faqs')->insertGetId($request->toArray());
            if ($insertedID) {
                $message = "FAQ added successfully";
                return $this->successResponse($message, ['faq_id' => $insertedID]);
            } else {
                return $this->errorResponse('Failed to add FAQ', self::STATUS_INTERNAL_SERVER_ERROR);
            }

        }

        else {
            return response()->json(['error' => 'Admin not found or does not belong to the authenticated admin'], 404);
        }

        return $this->errorResponse('Failed to add FAQ', self::STATUS_INTERNAL_SERVER_ERROR);


    }


    public function deleteFAQ(Request $request)
    {

        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }


        // Retrieve the user_type from the session
        if($session->user_type === "admin"){
            // Validate the incoming request
            $validator = Validator::make($request->all(), [
                'faq_id' => 'required|integer|exists:faqs,faqs_id',
            ]);

            // Return validation errors
            if ($validator->fails()) {
                return response()->json(['error' => $validator->errors()->first()], 400);
            }

            $faq = faq::find($request->faq_id);

            if (!$faq) {
                return response()->json(['message' => 'FAQ not found'], 404);
            }

            $faq->delete();

            return response()->json(['message' => 'FAQ deleted successfully'], 200);
        }

        else {
            return response()->json(['error' => 'Admin not found or does not belong to the authenticated admin'], 404);
        }

        return $this->errorResponse('Failed to delete FAQ', self::STATUS_INTERNAL_SERVER_ERROR);


    }

    public function getAllFAQ(Request $request)
    {
            $faqs = faq::orderBy('created_at', 'desc')->get();
            return response()->json($faqs);
    }


    public function getAllComplaints(Request $request)
    {

        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }


        // Retrieve the user_type from the session
        if($session->user_type === "admin"){
            $complaints = Complaints::orderBy('created_at', 'desc')->get();
            return response()->json($complaints);
        }

        else {
            return response()->json(['error' => 'Admin not found or does not belong to the authenticated admin'], 404);
        }

        return $this->errorResponse('Failed to fetch all complaints', self::STATUS_INTERNAL_SERVER_ERROR);

    }



    public function replyComplaint(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }


        // Retrieve the user_type from the session
        if($session->user_type === "admin"){

            // Validate the incoming request
            $validator = Validator::make($request->all(), [
                'complaint_id' => 'required|integer|exists:complaints,complaint_id',
                'reply' => 'required|string',
            ]);

            // Return validation errors
            if ($validator->fails()) {
                return response()->json(['error' => $validator->errors()->first()], 400);
            }

            $complaint = Complaints::find($request->complaint_id);

            if (!$complaint) {
                return response()->json(['message' => 'Complaint not found'], 404);
            }

            $complaint->update(['reply' => $request['reply']]);

            return response()->json(['message' => 'Reply added successfully', 'complaint' => $complaint]);
        }

        else {
            return response()->json(['error' => 'Admin not found or does not belong to the authenticated admin'], 404);
        }

        return $this->errorResponse('Failed to reply complaint', self::STATUS_INTERNAL_SERVER_ERROR);


    }



    public function getCoachDetailsWithClients(Request $request)
    {
        // Extract the token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided'
            ], 401);
        }

        // Find the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found'
            ], 401);
        }

        // Verify if the user is an admin
        if ($session->user_type !== 'admin') {
            return response()->json([
                'status' => 'error',
                'message' => 'Access denied. Only admins can access this API.'
            ], 403);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|integer|exists:coaches,coach_id',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Get the coach_id from the request
        $coachId = $request->input('coach_id');

        // Fetch the coach details
        $coach = Coach::find($coachId);

        if (!$coach) {
            return response()->json([
                'status' => 'error',
                'message' => 'Coach not found'
            ], 404);
        }

        // Fetch the latest subscription status for the coach
        $subscription = Subscription::where('coach_id', $coachId)
            ->orderBy('created_at', 'desc')
            ->first();

        $coach->subscription_status = $subscription ? $subscription->status : 'inactive';

        // Fetch all clients associated with the coach
        $clients = Clients::where('coach_id', $coachId)->orderBy('created_at', 'desc')->get();

        // Attach each client's coach name and subscription status
        $clients = $clients->map(function ($client) {
            // Attach coach name
            $coachhh = Coach::find($client->coach_id);
            $client->coach_name = $coachhh ? $coachhh->full_name : null;

            // Fetch subscription status based on the coach_id
            $subscription = Subscription::where('coach_id', $client->coach_id)
                ->orderBy('created_at', 'desc')
                ->first();

            $client->subscription_status = $subscription ? $subscription->status : 'inactive';

            return $client;
        });

        return response()->json([
            'status' => 'success',
            'coach' => $coach,
            'clients' => $clients
        ], 200);
    }









    public function registerSocialCoach(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'username' => 'required|string|max:255|unique:coaches', 
            'full_name' => 'required|string|max:255',             
            'phone' => 'required|string|max:255',  
            'email' => 'required|string|email|max:255|unique:coaches',             
            'city' => 'required|string|max:255',    
            'postal_code' => 'required|string|max:255',    
            'country' => 'required|string|max:255',

            'provider' => 'required|string|max:255',
            'social_id' => 'required|string|max:255',
        ]);

        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }


        // Prepare data for insertion
        $coachData = $request->only([
            'email', 
            'full_name', 
            'username', 
            'city', 
            'postal_code', 
            'country',  
            'is_email_verified',
            'profile_picture',
            'stripe_id',
            ]);

        $socialCoachData = $request->only([
            'provider',
            'social_id', 
            'coach_id',
            ]);

        // Email already verified social signup
        $coachData['is_email_verified'] = 1;

        // Set default profile picture if not provided
        $coachData['profile_picture'] = '/storage/images/profile_picture_default.png';



        // Initialize Stripe with your secret key
        $stripe = new StripeClient(env('STRIPE_SECRET_KEY'));

        // Create customer in Stripe
        $stripeCustomer = $stripe->customers->create([
            'email' => $request['email'],
            'name' => $request['full_name'],
            'phone' => $request['phone'],
            'address' => [
                'city' => $request['city'],
                'country' => $request['country'],
                'postal_code' => $request['postal_code'],
            ],
        ]);

        $coachData['stripe_id']  = $stripeCustomer->id;


        // Insert the coach and get the inserted ID
        $insertedCoachID = DB::table('coaches')->insertGetId($coachData);

        if ($insertedCoachID) {
            $coach = Coach::where('email', $request->email)->first();

            $socialCoachData['coach_id'] = $coach->coach_id;
            $insertedSocialCoachID = DB::table('social_coaches')->insertGetId($socialCoachData);

            // Extract device_info from User-Agent header
            $device_info = $request->header('User-Agent');
            
            // Concatenating IDs and encrypting
            $idsPayload = "{$coach->coach_id}-coach";
            $encryptedIds = encrypt($idsPayload);

            // Option A: Allow Multiple Sessions per coach
            // Create a new session record
            $session = new Session([
                // 'device_info' => $request->device_info,
                'device_info' => $device_info, // Use extracted device_info from headers
                'authentication_token' => $encryptedIds,
                'user_id' => $coach->coach_id,
                'user_type' => "coach"
            ]);
            $session->save();

            // Retrieve the full session from the database to include all fields
            $fullSession = Session::find($session->session_id);

            $response = [
                "success" => 1,
                "message" => "Coach registered successfully!",
                'session' => $fullSession->toArray(), // Returning the full session data
                "coach" => $coach->toArray(), // Sending as array for consistency
            ];
            return response($response, self::STATUS_SUCCESS);

        } else {
            return $this->errorResponse('Failed to register coach', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }


    public function socialCoachSignIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|max:255', 
            'provider' => 'required|string|max:255', 
            'social_id' => 'required|string|max:255', 
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Extract device_info from User-Agent header
        $device_info = $request->header('User-Agent');

        // Attempt to retrieve the social coach by email
        $coach = Coach::where('email', $request->email)->first();

        // Check if the coach exists in the database
        if (!$coach) {
            return response(['email_exists' => 0, 'message' => "Coach email doesn't exist"], self::STATUS_NOT_FOUND);
        }

        // Concatenating IDs and encrypting
        $idsPayload = "{$coach->coach_id}-coach";
        $encryptedIds = encrypt($idsPayload);
        $user_type  = "coach";


        // Option A: Allow Multiple Sessions per Coach
        // Create a new session record
        $session = new Session([
            // 'device_info' => $request->device_info,
            'device_info' => $device_info, // Use extracted device_info from headers
            'authentication_token' => $encryptedIds,
            'user_id' => $coach->coach_id,
            'user_type' => $user_type
        ]);
        $session->save();

        // Retrieve the full session from the database to include all fields
        $fullSession = Session::find($session->session_id);

        $response = [
            "success" => 1,
            'session' => $fullSession->toArray(), // Returning the full session data
            "coach" => $coach->toArray(), // Sending as array for consistency
        ];


        return response($response, self::STATUS_SUCCESS);
    }


















}






