<?php

namespace App\Http\Controllers; 
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use App\Helpers\NotificationHelper; 
use App\Services\FoodService;



use App\Models\User;
use App\Models\Coach;
use App\Models\Clients;
use App\Models\SocialClients;
use App\Models\Session;
use App\Models\Subscription;
use App\Models\paymentMethod;
use App\Models\Invoice;
use App\Models\checkInDetails;
use App\Models\CheckInPictures;
use App\Models\macroPlans;
use App\Models\exercises;
use App\Models\workoutPlans;
use App\Models\workoutDays;
use App\Models\repsSets;



use App\Models\Meals;
use App\Models\MealItems;
use App\Models\dietDays;
use App\Models\dietPlans;
use App\Models\clientPreferences;

use App\Models\faq;
use App\Models\Complaints;

use App\Models\RecentCoaches;
use App\Models\FavoriteCoaches;

use App\Models\ClientVerificationCodes;


use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Carbon;

use Illuminate\Support\Facades\Storage;
use Stripe\StripeClient;
use Stripe\Event;
use Stripe\Webhook;
use Stripe\Exception\SignatureVerificationException;
use Illuminate\Support\Facades\Http;

use Stripe\Stripe;



use Illuminate\Support\Facades\Crypt;







class ClientAPIController extends Controller
{
    const STATUS_SUCCESS = 200;
    const STATUS_BAD_REQUEST = 400;
    const STATUS_UNAUTHORIZED= 401;
    const STATUS_UNAUTHORIZED_EMAIL= 403;
    const STATUS_NOT_FOUND = 404;
    const STATUS_CONFLICT = 409;
    const STATUS_STRIPE_NOT_FOUND = 422;
    const STATUS_INTERNAL_SERVER_ERROR = 500;
    // Define more status codes as needed

    protected $foodService;


    public function __construct(FoodService $foodService)
    {
        $this->foodService = $foodService;
    }



    private function errorResponse($message, $statusCode)
    {
        return response(['status' =>'error', 'message' => $message], $statusCode);
    }

    private function successResponse($message, $extraData = [])
    {
        $response = array_merge(['status' =>'success', 'message' => $message], $extraData);
        return response($response, self::STATUS_SUCCESS);
    }

    public function checkClientUsername(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'username' => 'required|string|max:255',
        ]);

        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $client = Clients::where('username', $request->username)->first();
        if ($client) {
            return $this->successResponse("Client username found", ["alreadyExists" => 1]);
        } else {
            return $this->successResponse("Client username doesn't exist", ["alreadyExists" => 0]);
        }
    }


    public function verifyClientEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Generate a random 4-digit code
        $verificationCode = random_int(1000, 9999);

        // Calculate expiration time (e.g., 10 minutes from now)
        $expirationTime = Carbon::now()->addMinutes(10);

        // Create a new verification record in the database
        $verification = ClientVerificationCodes::create([
            'email' => $request->email,
            'code_for_email_verification' => $verificationCode,
            'code_for_email_verification_expires_at' => $expirationTime,
        ]);

        // Send the verification code to the email
        try {
            Mail::send('emails.verification_code', [
                'email' => $request->email,
                'verificationCode' => $verificationCode,
            ], function ($message) use ($request) {
                $message->to($request->email)
                        ->subject("Let's Verify Your Email for The Coaching App");
            });

        } catch (\Exception $e) {
            // Log the detailed error
            \Log::error('Mail send error: ' . $e->getMessage());

            // Return an error response without deleting the verification record
            return $this->errorResponse('Failed to send verification code. Please try again.', self::STATUS_INTERNAL_SERVER_ERROR);
        }

        // Respond with the code for front-end verification
        return $this->successResponse('Verification code sent, your code will be expired within 10 minutes from now.');
    }

    public function verifyVerificationCodeforClientEmail(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255',
            'code' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Fetch the latest verification record for the given email
        $client = ClientVerificationCodes::where('email', $request->email)
                                         ->orderBy('created_at', 'desc')
                                         ->first();

        if (!$client) {
            return $this->errorResponse('Email not found', self::STATUS_NOT_FOUND);
        }

        // Check if the verification code matches
        if ($client->code_for_email_verification != $request->code) {
            return $this->errorResponse('Invalid verification code', self::STATUS_BAD_REQUEST);
        }

        // Check if the verification code has expired
        if ($client->code_for_email_verification_expires_at < Carbon::now()) {
            return $this->errorResponse('Verification code has expired', self::STATUS_CONFLICT);
        }

        // Verification successful
        // Do not clear the verification code and expiration time
        return $this->successResponse('Verification successful.');
    }


    public function verifyClientForgetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Find the client by email
        $client = Clients::where('email', $request->email)->first();
        if (!$client) {
            return $this->errorResponse('Client not found', self::STATUS_NOT_FOUND);
        }

        // Generate a random 4-digit code
        $verificationCode = random_int(1000, 9999);

        // Calculate expiration time (e.g., 10 minutes from now)
        $expirationTime = Carbon::now()->addMinutes(10);

        // Send the verification code to the email
        try {
            Mail::send('emails.forgot_password_verification', [
                'email' => $request->email,
                'verificationCode' => $verificationCode,
            ], function ($message) use ($request) {
                $message->to($request->email)
                        ->subject("Let's Verify Your Email to Reset Your Password");
            });

            // Store the verification code and expiration time in the database
            $client->code_for_forgot_password_verification = $verificationCode;
            $client->code_for_forgot_password_verification_expires_at = $expirationTime;
            $client->save();

        } catch (\Exception $e) {
            // Log the detailed error
            \Log::error('Mail send error: ' . $e->getMessage());

            // Return an error response
            return $this->errorResponse('Failed to send verification code.', self::STATUS_INTERNAL_SERVER_ERROR);
        }

        // Respond with the code for front-end verification
        return $this->successResponse('Verification code sent, your code will be expired within 10 minutes from now.');
    }

    public function verifyVerificationCodeforClientForgetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255',
            'code' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $client = Clients::where('email', $request->email)->first();
        if (!$client) {
            return $this->errorResponse('Client not found', self::STATUS_NOT_FOUND);
        }

        if (!$client || $client->code_for_forgot_password_verification != $request->code) {
            return $this->errorResponse('Invalid verification code', self::STATUS_BAD_REQUEST);
        }

        // Use Carbon to ensure consistent time comparison
        $currentTime = Carbon::now();
        $expirationTime = Carbon::parse($client->code_for_forgot_password_verification_expires_at);

        // Check if the verification code has expired
        if ($currentTime->gt($expirationTime)) {
            return $this->errorResponse('Verification code has expired', self::STATUS_BAD_REQUEST);
        }

        // Verification successful
        // Clear the verification code and expiration time from the database
        $client->code_for_forgot_password_verification = null;
        $client->code_for_forgot_password_verification_expires_at = null;
        $client->save();

        return $this->successResponse('Verification successful');
    }

    public function updateClientPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|max:255',
            'new_password' => 'required|string|max:255',                         
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Fetch the client from the database
        $client = Clients::where('email', $request->email)->first();
        if (!$client) {
            return $this->errorResponse('Client not found', self::STATUS_NOT_FOUND);
        }

        $client->password = Hash::make($request->new_password);
        $updateSuccess = $client->save(); // Save the updated password

        if ($updateSuccess) {
            return $this->successResponse('Password updated successfully');
        } else {
            return $this->errorResponse('Failed to update password', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }

    public function searchByUsername(Request $request)
    {

        $validator = Validator::make($request->all(), [
                'username' => 'required|string|min:1|max:255',
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        $username = strtolower($request->username);
        // $coaches = Coach::whereRaw('LOWER(username) LIKE ?', [$username . '%'])->get();

        // If the requirement is to search for coaches whose usernames contain the provided username parameter anywhere (not just at the beginning like the above line of query), you can modify the query to use the LIKE operator with % on both sides of the search term. This will allow matching the search term anywhere in the username. 
        $coaches = Coach::whereRaw('LOWER(username) LIKE ?', ['%' . $username . '%'])->with('activeClientPreferences')->get();

        // Return the response
        $response = [
            'status' => 'Success',
            'coaches' => $coaches,
        ];

        return response($response, self::STATUS_SUCCESS);
    }


    // Always verified client will register, based on Umer's explanation on whatsapp
    // public function registerClient(Request $request)
    // {
    //     $validator = Validator::make($request->all(), [ 
    //         'email' => 'required|string|email|max:255|unique:clients',   
    //         'password' => 'required|string|max:255',
    //         'full_name' => 'required|string|max:255', 
    //         'username' => 'required|string|max:255|unique:clients', 

    //         'gender' => 'nullable|max:255', 
    //         'age' => 'nullable|max:255',    
    //         'level' => 'nullable|max:255',    
    //         'weight' => 'nullable|max:255',
    //         'height' => 'nullable|max:255',
    //         'coach_id' => 'nullable|max:255|exists:coaches,coach_id',
    //     ]);

    //     if ($validator->fails())
    //     {
    //         return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
    //     }

    //     // Prepare data for insertion
    //     $data = $request->only([
    //         'email', 
    //         'password', 
    //         'full_name', 
    //         'username', 
    //         'gender', 
    //         'age', 
    //         'level', 
    //         'weight', 
    //         'height', 
    //         'coach_id',
    //         'is_email_verified',
    //         ]);

    //     // Hash the password
    //     $data['password'] = Hash::make($data['password']);

    //     // Email already verified first before registration (from the app)
    //     $data['is_email_verified'] = 1;

    //     // Insert the client and get the inserted ID
    //     $insertedID = DB::table('clients')->insertGetId($data);

    //     if ($insertedID) {
    //         $client = Clients::find($insertedID);

    //         // Extract device_info from User-Agent header
    //         $device_info = $request->header('User-Agent');
            
    //         // Concatenating IDs and encrypting
    //         $idsPayload = "{$client->client_id}-client";
    //         $encryptedIds = encrypt($idsPayload);

    //         // Option A: Allow Multiple Sessions per Client
    //         // Create a new session record
    //         $session = new Session([
    //             // 'device_info' => $request->device_info,
    //             'device_info' => $device_info, // Use extracted device_info from headers
    //             'authentication_token' => $encryptedIds,
    //             'user_id' => $client->client_id,
    //             'user_type' => "client"
    //         ]);
    //         $session->save();

    //         // Retrieve the full session from the database to include all fields
    //         $fullSession = Session::find($session->session_id);

    //         $response = [
    //             "success" => 1,
    //             'session' => $fullSession->toArray(), // Returning the full session data
    //             "client" => $client->toArray(), // Sending as array for consistency

    //         ];
    //         return response($response, self::STATUS_SUCCESS);

    //     } else {
    //         return $this->errorResponse('Failed to register client', self::STATUS_INTERNAL_SERVER_ERROR);
    //     }
    // }


    public function registerClient(Request $request)
    {
        $validator = Validator::make($request->all(), [ 
            'email' => [
                        'required',
                        'string',
                        'email',
                        'max:255',
                        'unique:clients',
                        function ($attribute, $value, $fail) {
                            if (DB::table('coaches')->where('email', $value)->exists()) {
                                $fail('This email is already registered as a coach.');
                            }
                        },
                    ],
            'password' => 'required|string|max:255',
            'full_name' => 'required|string|max:255', 
            'username' => 'required|string|max:255|unique:clients', 

            'gender' => 'nullable|max:255', 
            'age' => 'nullable|max:255',    
            'level' => 'nullable|max:255',    
            'weight' => 'nullable|max:255',
            'height' => 'nullable|max:255',
            'coach_id' => 'nullable|max:255|exists:coaches,coach_id',
        ]);

        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Prepare data for insertion
        $data = $request->only([
            'email', 
            'password', 
            'full_name', 
            'username', 
            'gender', 
            'age', 
            'level', 
            'weight', 
            'height', 
            'coach_id',
            'is_email_verified',
            ]);

        // Hash the password
        $data['password'] = Hash::make($data['password']);

        // Email already verified first before registration (from the app)
        $data['is_email_verified'] = 1;

        // Set default profile picture if not provided
        $data['profile_picture'] = '/storage/images/profile_picture_default.png';

        // Insert the client and get the inserted ID
        $insertedID = DB::table('clients')->insertGetId($data);

        if ($insertedID) {
            $client = Clients::find($insertedID);

            // Extract device_info from User-Agent header
            $device_info = $request->header('User-Agent');
            
            // Concatenating IDs and encrypting
            $idsPayload = "{$client->client_id}-client";
            $encryptedIds = encrypt($idsPayload);

            // Option A: Allow Multiple Sessions per Client
            // Create a new session record
            $session = new Session([
                // 'device_info' => $request->device_info,
                'device_info' => $device_info, // Use extracted device_info from headers
                'authentication_token' => $encryptedIds,
                'user_id' => $client->client_id,
                'user_type' => "client"
            ]);
            $session->save();

            // Retrieve the full session from the database to include all fields
            $fullSession = Session::find($session->session_id);

            $response = [
                "success" => 1,
                'session' => $fullSession->toArray(), // Returning the full session data
                "client" => $client->toArray(), // Sending as array for consistency

            ];
            return response($response, self::STATUS_SUCCESS);

        } else {
            return $this->errorResponse('Failed to register client', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }


    public function clientSignIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email_username' => 'required|string|max:255', 
            'password' => 'required|string|max:255', 
            // 'device_info' => 'required|string|max:255', 
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Extract device_info from User-Agent header
        $device_info = $request->header('User-Agent');

        // Attempt to retrieve the client by email or username
        $client = Clients::where('email', $request->email_username)
                      ->orWhere('username', $request->email_username)->first();

        // Check if the client exists in the database
        if (!$client) {
            return $this->errorResponse('No client found', self::STATUS_NOT_FOUND);
        }
        
        // return $this->errorResponse(['Received Password Plain Text'=> $request->password, 'Received Password Hash'=> Hash::make($request->password), "Client Password"=>$client->password, 'Password Comparison: ' => (Hash::check($request->password, $client->password) ? 'Match' : 'No Match')], self::STATUS_UNAUTHORIZED);

        // Verify the password is correct
        if (!Hash::check($request->password, $client->password)) {
            return $this->errorResponse('Incorrect password', self::STATUS_UNAUTHORIZED);
        }

        // Check if the email is verified
        if (!$client->is_email_verified) {
            return $this->errorResponse('Email is not verified', self::STATUS_UNAUTHORIZED_EMAIL);
        }

        // Concatenating IDs and encrypting
        $idsPayload = "{$client->client_id}-client";
        $encryptedIds = encrypt($idsPayload);
        $user_type  = "client";


        // Option A: Allow Multiple Sessions per Coach
        // Create a new session record
        $session = new Session([
            // 'device_info' => $request->device_info,
            'device_info' => $device_info, // Use extracted device_info from headers
            'authentication_token' => $encryptedIds,
            'user_id' => $client->client_id,
            'user_type' => $user_type
        ]);
        $session->save();

        // Retrieve the full session from the database to include all fields
        $fullSession = Session::find($session->session_id);

        $response = [
            "success" => 1,
            'session' => $fullSession->toArray(), // Returning the full session data
            "client" => $client->toArray(), // Sending as array for consistency
        ];


        return response($response, self::STATUS_SUCCESS);
    }


    public function updateFcmToken(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'fcm_token' => 'required|string',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$userId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $userId = $session->user_id;

            // Validate the user_id and user_type
            if ($session->user_id != $userId || $session->user_type !== 'client') {
                return $this->errorResponse('Token does not match the session or user type is not client', self::STATUS_UNAUTHORIZED_EMAIL);
            }
        } catch (\Exception $e) {
            return $this->errorResponse('Invalid token format', self::STATUS_BAD_REQUEST);
        }

        // Update the FCM token
        $session->fcm_token = $request->fcm_token;
        $session->save();

        // Return a success response
        return response()->json([
            'status' => 'success',
            'message' => 'FCM token updated successfully',
        ]);
    }


    // Always verified client will register, based on Umer's explanation on whatsapp
    public function registerSocialClient(Request $request)
    {
        $validator = Validator::make($request->all(), [ 
            'email' => [
                        'required',
                        'string',
                        'email',
                        'max:255',
                        'unique:clients',
                        function ($attribute, $value, $fail) {
                            if (DB::table('coaches')->where('email', $value)->exists()) {
                                $fail('This email is already registered as a coach.');
                            }
                        },
                    ],
            'full_name' => 'required|string|max:255', 
            'username' => 'required|string|max:255|unique:clients', 
            'provider' => 'required|string|max:255', 
            'social_id' => 'required|string|max:255',

            'gender' => 'nullable|max:255', 
            'age' => 'nullable|max:255',    
            'level' => 'nullable|max:255',    
            'weight' => 'nullable|max:255',
            'height' => 'nullable|max:255',
            'coach_id' => 'nullable|max:255|exists:coaches,coach_id',
        ]);

        if ($validator->fails())
        {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Prepare data for insertion
        $clientData = $request->only([
            'email', 
            'full_name', 
            'username', 
            'gender', 
            'age', 
            'level', 
            'weight', 
            'height', 
            'coach_id',
            'is_email_verified',
            ]);

        $socialClientData = $request->only([
            'provider',
            'social_id', 
            'client_id',
            ]);

        // Email already verified first before registration (from the app)
        $clientData['is_email_verified'] = 1;

        // Insert the client and get the inserted ID
        $insertedClientID = DB::table('clients')->insertGetId($clientData);

        if ($insertedClientID) {
            $client = Clients::find($insertedClientID);

            $socialClientData['client_id'] = $client->client_id;
            $insertedSocialClientID = DB::table('social_clients')->insertGetId($socialClientData);

            // Extract device_info from User-Agent header
            $device_info = $request->header('User-Agent');
            
            // Concatenating IDs and encrypting
            $idsPayload = "{$client->client_id}-client";
            $encryptedIds = encrypt($idsPayload);

            // Option A: Allow Multiple Sessions per Client
            // Create a new session record
            $session = new Session([
                // 'device_info' => $request->device_info,
                'device_info' => $device_info, // Use extracted device_info from headers
                'authentication_token' => $encryptedIds,
                'user_id' => $client->client_id,
                'user_type' => "client"
            ]);
            $session->save();

            // Retrieve the full session from the database to include all fields
            $fullSession = Session::find($session->session_id);

            $response = [
                "success" => 1,
                'session' => $fullSession->toArray(), // Returning the full session data
                "client" => $client->toArray(), // Sending as array for consistency
            ];
            return response($response, self::STATUS_SUCCESS);

        } else {
            return $this->errorResponse('Failed to register client', self::STATUS_INTERNAL_SERVER_ERROR);
        }
    }

    public function socialClientSignIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|max:255', 
            'provider' => 'required|string|max:255', 
            'social_id' => 'required|string|max:255', 
        ]);

        if ($validator->fails()) {
            return $this->errorResponse($validator->errors()->first(), self::STATUS_BAD_REQUEST);
        }

        // Extract device_info from User-Agent header
        $device_info = $request->header('User-Agent');

        // Attempt to retrieve the social client by email
        $client = Clients::where('email', $request->email)->first();

        // Check if the client exists in the database
        if (!$client) {
            return response(['email_exists' => 0, 'message' => "Client email doesn't exist"], self::STATUS_NOT_FOUND);
        }

        // Concatenating IDs and encrypting
        $idsPayload = "{$client->client_id}-client";
        $encryptedIds = encrypt($idsPayload);
        $user_type  = "client";


        // Option A: Allow Multiple Sessions per Coach
        // Create a new session record
        $session = new Session([
            // 'device_info' => $request->device_info,
            'device_info' => $device_info, // Use extracted device_info from headers
            'authentication_token' => $encryptedIds,
            'user_id' => $client->client_id,
            'user_type' => $user_type
        ]);
        $session->save();

        // Retrieve the full session from the database to include all fields
        $fullSession = Session::find($session->session_id);

        $response = [
            "success" => 1,
            'session' => $fullSession->toArray(), // Returning the full session data
            "client" => $client->toArray(), // Sending as array for consistency
        ];


        return response($response, self::STATUS_SUCCESS);
    }







    public function favoriteCoach(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255',
        ]);

        // Return validation errors
        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Attempt to retrieve the coach by coach_id
        $coach = Coach::where('coach_id', $request->coach_id)->first();

        // Check if the coach exists in the database
        if (!$coach) {
            return $this->errorResponse('No coach found', self::STATUS_NOT_FOUND);
        }


        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;

            // Validate the user_id and user_type
            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Check if the favorite record already exists
        $exists = FavoriteCoaches::where('client_id', $clientId)
            ->where('coach_id', $request->coach_id)
            ->exists();

        if ($exists) {
            return response()->json([
                'status' => 'error',
                'message' => 'Coach already favorited',
            ], 400);
        }

        // Try to add the coach to the favorites
        try {
            FavoriteCoaches::create([
                'client_id' => $clientId,
                'coach_id' => $request->coach_id,
            ]);

            // Return success response
            return response()->json([
                'status' => 'success',
                'message' => 'Coach favorited successfully',
            ], 200);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error favoriting coach:', ['error' => $e->getMessage()]);

            // Return error response
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to favorite coach. Please try again later.',
            ], 500);
        }
    }

    public function fetchFavoriteCoaches(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;

            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Fetch all favorite coaches for the client
        try {
            $favoriteCoaches = FavoriteCoaches::with('coach') // Assuming there's a `coach` relationship in FavoriteCoach
                ->where('client_id', $clientId)
                ->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Favorite coaches fetched successfully',
                'data' => $favoriteCoaches,
            ], 200);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error fetching favorite coaches:', ['error' => $e->getMessage()]);

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch favorite coaches. Please try again later.',
            ], 500);
        }
    }

    public function removeFavoriteCoach(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Attempt to retrieve the coach by coach_id
        $coach = Coach::where('coach_id', $request->coach_id)->first();

        // Check if the coach exists in the database
        if (!$coach) {
            return $this->errorResponse('No coach found', self::STATUS_NOT_FOUND);
        }


        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;

            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Try to remove the favorite coach
        try {
            $deleted = FavoriteCoaches::where('client_id', $clientId)
                ->where('coach_id', $request->coach_id)
                ->delete();

            if ($deleted) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'Coach removed from favorites successfully',
                ], 200);
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Favorite coach not found',
                ], 404);
            }
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error removing favorite coach:', ['error' => $e->getMessage()]);

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to remove favorite coach. Please try again later.',
            ], 500);
        }
    }


    // public function fetchRecentCoaches(Request $request)
    // {
    //     // Extract the authentication token from the Authorization header
    //     $token = $request->bearerToken();

    //     if (!$token) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Token not provided',
    //         ], 401);
    //     }

    //     // Retrieve the session based on the provided authentication token
    //     $session = Session::where('authentication_token', $token)->first();

    //     if (!$session) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token or session not found',
    //         ], 401);
    //     }

    //     // Decrypt the authentication token to extract user_id and user_type
    //     try {
    //         $decryptedToken = decrypt($token); // Example: "123-client"
    //         [$clientId, $userType] = explode('-', $decryptedToken);

    //         // or we can just use this to retrieve the client_id from the session:
    //         // $clientId = $session->user_id;

    //         if ($session->user_id != $clientId || $session->user_type !== 'client') {
    //             return response()->json([
    //                 'status' => 'error',
    //                 'message' => 'Token does not match the session or user type is not client',
    //             ], 401);
    //         }
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token format',
    //         ], 400);
    //     }

    //     // Fetch recent coaches for the client
    //     try {
    //         $recentCoaches = RecentCoaches::with('coach') // Assuming there's a `coach` relationship in RecentCoach
    //             ->where('client_id', $clientId)
    //             ->orderBy('created_at', 'desc')
    //             ->get();

    //         return response()->json([
    //             'status' => 'success',
    //             'message' => 'Recent coaches fetched successfully',
    //             'recent_coaches' => $recentCoaches,
    //         ], 200);
    //     } catch (\Exception $e) {
    //         // Log the error for debugging
    //         \Log::error('Error fetching recent coaches:', ['error' => $e->getMessage()]);

    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Failed to fetch recent coaches. Please try again later.',
    //         ], 500);
    //     }
    // }
    
    public function fetchRecentCoaches(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$clientId, $userType] = explode('-', $decryptedToken);

            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Fetch recent coaches for the client
        try {
            $recentCoaches = RecentCoaches::with('coach') // Assuming there's a `coach` relationship in RecentCoach
                ->where('client_id', $clientId)
                ->orderBy('created_at', 'desc')
                ->get();

            // Add isFav and isPref to each coach in the recent coaches list
            foreach ($recentCoaches as &$coachRecord) {
                $coach = $coachRecord->coach; // Assuming the coach relationship exists

                // Check if the client has marked this coach as a favorite
                $coach->isFav = FavoriteCoaches::where('client_id', $clientId)
                    ->where('coach_id', $coach->coach_id)
                    ->exists();

                // Check if the client has added this coach to their preferences
                $preference = clientPreferences::where('client_id', $clientId)
                    ->where('coach_id', $coach->coach_id)
                    ->first();

                if ($preference) {
                    // Check the status of the preference
                    if (
                        ($preference->status_by_client === 'rejected' || $preference->status_by_coach === 'rejected') ||
                        ($preference->status_by_client === 'accepted' && $preference->status_by_coach === 'rejected') ||
                        ($preference->status_by_coach === 'accepted' && $preference->status_by_client === 'rejected')
                    ) {
                        $coach->isPref = false;
                    } else {
                        $coach->isPref = true;
                    }
                } else {
                    $coach->isPref = true; // Default to true if no preference exists
                }
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Recent coaches fetched successfully',
                'recent_coaches' => $recentCoaches,
            ], 200);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error fetching recent coaches:', ['error' => $e->getMessage()]);

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch recent coaches. Please try again later.',
            ], 500);
        }
    }


    public function addCoachPreference(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Attempt to retrieve the coach by coach_id
        $coach = Coach::where('coach_id', $request->coach_id)->first();

        // Check if the coach exists in the database
        if (!$coach) {
            return $this->errorResponse('No coach found', self::STATUS_NOT_FOUND);
        }


        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;

            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Add the coach preference for the client
        try {
            // Allow duplicates in the preferences table as the client can choose the same coach multiple times
            clientPreferences::create([
                'client_id' => $clientId,
                'coach_id' => $request->coach_id,
                'message' => 'looking for a coach', // The status set when adding the preference
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Coach preference added successfully',
            ], 200);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error adding coach preference:', ['error' => $e->getMessage()]);

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to add coach preference. Please try again later.',
            ], 500);
        }
    }



    // public function updateClientProfile(Request $request)
    // {
    //     // Extract the authentication token
    //     $token = $request->bearerToken();

    //     if (!$token) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Token not provided',
    //         ], 401);
    //     }

    //     // Retrieve the session based on the provided authentication token
    //     $session = Session::where('authentication_token', $token)->first();

    //     if (!$session) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token or session not found',
    //         ], 401);
    //     }

    //     // Decrypt the authentication token to extract client_id
    //     try {
    //         $decryptedToken = decrypt($token);
    //         [$userId, $userType] = explode('-', $decryptedToken);

    //         // or we can just use this to retrieve the client_id from the session:
    //         // $userId = $session->user_id;

    //         // Validate the user_id and user_type
    //         if ($session->user_id != $userId || $session->user_type !== 'client') {
    //             return response()->json([
    //                 'status' => 'error',
    //                 'message' => 'Token does not match the session or user type is not client',
    //             ], 401);
    //         }
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Invalid token format',
    //         ], 400);
    //     }

    //     // Validate the incoming request data
    //     $validator = Validator::make($request->all(), [
    //         'profile_picture' => 'image|mimes:jpeg,png,jpg,gif',
    //         'gender' => 'string|max:255',
    //         'age' => 'string|max:255',
    //         'level' => 'string|max:255',
    //         'weight' => 'string|max:255',
    //         'height' => 'string|max:255',
    //     ]);

    //     if ($validator->fails()) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => $validator->errors()->first(),
    //         ], 400);
    //     }

    //     // Retrieve the client using the extracted client_id
    //     $client = Clients::find($userId);

    //     if (!$client) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Client not found',
    //         ], 404);
    //     }

    //     // Handle file uploads (profile picture)
    //     if ($request->hasFile('profile_picture')) {
    //         // Delete the previous profile picture if it exists
    //         if ($client->profile_picture) {
    //             Storage::delete(str_replace('/storage', 'public', $client->profile_picture));
    //         }

    //         // Handle the file upload for the profile picture
    //         $filename = $request->file('profile_picture')->getClientOriginalName();
    //         $fileExtension = $request->file('profile_picture')->getClientOriginalExtension();
    //         $newFileName = "profile_picture_" . time() . '.' . $fileExtension;
    //         $img_path = $request->file('profile_picture')->storeAs('public/images', $newFileName);

    //         // Save the new file path to the client record
    //         $client->profile_picture = '/storage/images/' . $newFileName;
    //     }

    //     // Update other client fields
    //     $client->gender = $request->gender;
    //     $client->age = $request->age;
    //     $client->level = $request->level;
    //     $client->weight = $request->weight;
    //     $client->height = $request->height;

    //     // Save the updated client profile
    //     try {
    //         $client->save();

    //         return response()->json([
    //             'status' => 'success',
    //             'message' => 'Profile updated successfully',
    //             'data' => $client
    //         ], 200);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'Failed to update profile: ' . $e->getMessage(),
    //         ], 500);
    //     }
    // }

    public function updateClientProfile(Request $request)
    {
        // Extract the authentication token
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract client_id
        try {
            $decryptedToken = decrypt($token);
            [$userId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $userId = $session->user_id;

            // Validate the user_id and user_type
            if ($session->user_id != $userId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Validate the incoming request data
        $validator = Validator::make($request->all(), [
            'profile_picture' => 'nullable|image|mimes:jpeg,png,jpg,gif',
            'gender' => 'nullable|string|max:255',
            'age' => 'nullable|string|max:255',
            'level' => 'nullable|string|max:255',
            'weight' => 'nullable|string|max:255',
            'height' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first(),
            ], 400);
        }

        // Retrieve the client using the extracted client_id
        $client = Clients::find($userId);

        if (!$client) {
            return response()->json([
                'status' => 'error',
                'message' => 'Client not found',
            ], 404);
        }

        // Handle file uploads (profile picture)
        if ($request->hasFile('profile_picture')) {
            // Delete the previous profile picture if it exists
            if ($client->profile_picture) {
                Storage::delete(str_replace('/storage', 'public', $client->profile_picture));
            }

            // Handle the file upload for the profile picture
            $filename = $request->file('profile_picture')->getClientOriginalName();
            $fileExtension = $request->file('profile_picture')->getClientOriginalExtension();
            $newFileName = "profile_picture_" . time() . '.' . $fileExtension;
            $img_path = $request->file('profile_picture')->storeAs('public/images', $newFileName);

            // Save the new file path to the client record
            $client->profile_picture = '/storage/images/' . $newFileName;
        } else {
            if (!$client->profile_picture) {
                $client->profile_picture = '/storage/images/profile_picture_default.png';
            }
        }

        // Update other client fields
        $client->gender = $request->gender;
        $client->age = $request->age;
        $client->level = $request->level;
        $client->weight = $request->weight;
        $client->height = $request->height;

        // Save the updated client profile
        try {
            $client->save();

            return response()->json([
                'status' => 'success',
                'message' => 'Profile updated successfully',
                'data' => $client
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update profile: ' . $e->getMessage(),
            ], 500);
        }
    }




    public function fetchAllAssignedPlans(Request $request)
    {
        // Extract the authentication token
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract client_id
        try {
            $decryptedToken = decrypt($token);
            [$userId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $userId = $session->user_id;

            // Validate the user_id and user_type
            if ($session->user_id != $userId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Validate the incoming request data
        $validator = Validator::make($request->all(), [
            'coach_id' => 'nullable|string|max:255',
        ]);

        // Return validation errors if any
        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first(),
            ], 400);
        }

        // Retrieve the client using the extracted client_id
        $client = Clients::find($userId);

        if (!$client) {
            return response()->json([
                'status' => 'error',
                'message' => 'Client not found',
            ], 404);
        }


        // If coach_id is not provided, use the current coach assigned to the client
        $coach_id = $request->coach_id ?? $client->coach_id;

        // Fetch the client's assigned plans
        $macro_plans = macroPlans::where('client_assigned_id', $userId)
                                ->where('coach_id', $coach_id)
                                ->orderBy('created_at', 'desc')
                                ->get();

        $workout_plans = workoutPlans::where('client_assigned_id', $userId)
                                    ->where('coach_id', $coach_id)
                                    ->with(['workoutDays.exercises.repsSets'])
                                    ->orderBy('created_at', 'desc')
                                    ->get();

        $diet_plans = dietPlans::where('assigned_to', $userId)
                              ->where('coach_id', $coach_id)
                              ->with(['dietDays.meals.mealItems'])
                              ->orderBy('created_at', 'desc')
                              ->get();

        // Construct the response
        $response = [
            'macro_plans' => $macro_plans,
            'workout_plans' => $workout_plans,
            'diet_plans' => $diet_plans,
        ];

        return response()->json([
            'status' => 'success',
            'message' => 'Assigned plans fetched successfully',
            'data' => $response
        ], 200);
    }


    // public function getAllCoachesClient(Request $request)
    // {
    //     // Extract the authentication token
    //     $token = $request->bearerToken();
    //     if (!$token) {
    //         return response()->json(['status' => 'error', 'message' => 'Token not provided'], 401);
    //     }

    //     // Validate and decrypt token
    //     try {
    //         $decryptedToken = decrypt($token);
    //         [$clientId, $userType] = explode('-', $decryptedToken);

    //         // or we can just use this to retrieve the client_id from the session:
    //         // $clientId = $session->user_id;

    //         if ($userType !== 'client') {
    //             return response()->json(['status' => 'error', 'message' => 'Invalid user type'], 401);
    //         }
    //     } catch (\Exception $e) {
    //         return response()->json(['status' => 'error', 'message' => 'Invalid token format'], 400);
    //     }

    //     // Validate the incoming request
    //     $validator = Validator::make($request->all(), [
    //         'city' => 'required|string|max:255',
    //     ]);

    //     if ($validator->fails()) {
    //         return response()->json(['error' => $validator->errors()->first()], 400);
    //     }

    //     $city = strtolower($request->city);

    //     if (!$city) {
    //         return response()->json(['status' => 'error', 'message' => 'City parameter is required'], 400);
    //     }

    //     // Helper function to fetch nearby cities based on postal code
    //     $nearbyCities = $this->getNearbyCities($city);

    //     // Fetch IDs for preferences, favorites, and recent interactions
    //     $preferredCoachIds = DB::table('client_preferences')
    //         ->where('client_id', $clientId)
    //         ->where('client_status', 'accept')
    //         ->where('coach_status', 'accept')
    //         ->pluck('coach_id');
    //     $favoriteCoachIds = FavoriteCoaches::where('client_id', $clientId)->pluck('coach_id');
    //     $recentCoachIds = RecentCoaches::where('client_id', $clientId)->pluck('coach_id');

    //     // Combine interaction-related IDs
    //     $interactionCoachIds = $preferredCoachIds
    //         ->merge($favoriteCoachIds)
    //         ->merge($recentCoachIds)
    //         ->unique();

    //     // Fetch all coaches
    //     $allCoaches = Coach::query()
    //         ->withCount(['favorites', 'checkInDetails', 'recentInteractions']) // Include engagement counts
    //         ->get();

    //     $bestCoaches = collect();
    //     $topPicks = collect();

    //     // Define limits for balanced distribution
    //     $totalCoaches = $allCoaches->count();
    //     $halfLimit = floor($totalCoaches / 2);

    //     // Separate coaches into Top Picks and Best Coaches
    //     foreach ($allCoaches as $coach) {
    //         if (
    //             str_contains(strtolower($coach->city), $city) && // Exact city match
    //             $topPicks->count() < $halfLimit // Ensure Top Picks doesn't exceed the limit
    //         ) {
    //             $topPicks->push($coach);
    //         } elseif (
    //             $interactionCoachIds->contains($coach->coach_id) && // Interaction match
    //             $bestCoaches->count() < $halfLimit // Ensure Best Coaches doesn't exceed the limit
    //         ) {
    //             $bestCoaches->push($coach);
    //         } elseif (
    //             in_array($coach->city, $nearbyCities) && // Nearby city match
    //             $topPicks->count() < $halfLimit // Ensure Top Picks doesn't exceed the limit
    //         ) {
    //             $topPicks->push($coach);
    //         }
    //     }

    //     // Add remaining coaches to balance the lists
    //     $remainingCoaches = $allCoaches->diff($bestCoaches)->diff($topPicks);

    //     foreach ($remainingCoaches as $coach) {
    //         if ($topPicks->count() < $halfLimit) {
    //             $topPicks->push($coach);
    //         } else {
    //             $bestCoaches->push($coach);
    //         }
    //     }

    //     // Sort Top Picks
    //     $topPicks = $topPicks->sortByDesc(function ($coach) {
    //         return [$coach->favorites_count, $coach->recent_interactions_count, $coach->check_in_details_count];
    //     })->values();

    //     // Sort Best Coaches
    //     $bestCoaches = $bestCoaches->sortByDesc(function ($coach) {
    //         return [$coach->favorites_count, $coach->recent_interactions_count, $coach->check_in_details_count];
    //     })->values();

    //     // Return response
    //     return response()->json([
    //         'status' => 'success',
    //         'message' => 'Coaches fetched successfully',
    //         'data' => [
    //             'top_picks' => $topPicks,
    //             'best_coaches' => $bestCoaches,
    //         ],
    //     ]);

    // }


    // public function getAllCoachesClient(Request $request)
    // {
    //     // Extract the authentication token
    //     $token = $request->bearerToken();
    //     if (!$token) {
    //         return response()->json(['status' => 'error', 'message' => 'Token not provided'], 401);
    //     }

    //     // Validate and decrypt token
    //     try {
    //         $decryptedToken = decrypt($token);
    //         [$clientId, $userType] = explode('-', $decryptedToken);

    //         if ($userType !== 'client') {
    //             return response()->json(['status' => 'error', 'message' => 'Invalid user type'], 401);
    //         }
    //     } catch (\Exception $e) {
    //         return response()->json(['status' => 'error', 'message' => 'Invalid token format'], 400);
    //     }

    //     // Make the city parameter optional
    //     $validator = Validator::make($request->all(), [
    //         'city' => 'nullable|max:255',
    //     ]);

    //     if ($validator->fails()) {
    //         return response()->json(['status' => 'error', 'message' => $validator->errors()->first()], 400);
    //     }

    //     $city = strtolower($request->city);

    //     // Fetch IDs for preferences, favorites, and recent interactions
    //     $preferredCoachIds = DB::table('client_preferences')
    //         ->where('client_id', $clientId)
    //         ->where('client_status', 'accept')
    //         ->where('coach_status', 'accept')
    //         ->pluck('coach_id');
    //     $favoriteCoachIds = FavoriteCoaches::where('client_id', $clientId)->pluck('coach_id');
    //     $recentCoachIds = RecentCoaches::where('client_id', $clientId)->pluck('coach_id');

    //     // Combine interaction-related IDs
    //     $interactionCoachIds = $preferredCoachIds
    //         ->merge($favoriteCoachIds)
    //         ->merge($recentCoachIds)
    //         ->unique();

    //     // Fetch all coaches
    //     $allCoaches = Coach::query()
    //         ->withCount(['favorites', 'checkInDetails', 'recentInteractions']) // Include engagement counts
    //         ->get();

    //     $bestCoaches = collect();
    //     $topPicks = collect();

    //     // Define limits for balanced distribution
    //     $totalCoaches = $allCoaches->count();
    //     $halfLimit = floor($totalCoaches / 2);

    //     // If city is provided, filter coaches by city
    //     if ($city) {
    //         // Helper function to fetch nearby cities based on postal code
    //         $nearbyCities = $this->getNearbyCities($city);

    //         // Separate coaches into Top Picks and Best Coaches based on city
    //         foreach ($allCoaches as $coach) {
    //             if (
    //                 str_contains(strtolower($coach->city), $city) && // Exact city match
    //                 $topPicks->count() < $halfLimit // Ensure Top Picks doesn't exceed the limit
    //             ) {
    //                 $topPicks->push($coach);
    //             } elseif (
    //                 $interactionCoachIds->contains($coach->coach_id) && // Interaction match
    //                 $bestCoaches->count() < $halfLimit // Ensure Best Coaches doesn't exceed the limit
    //             ) {
    //                 $bestCoaches->push($coach);
    //             } elseif (
    //                 in_array($coach->city, $nearbyCities) && // Nearby city match
    //                 $topPicks->count() < $halfLimit // Ensure Top Picks doesn't exceed the limit
    //             ) {
    //                 $topPicks->push($coach);
    //             }
    //         }

    //     } else {
    //         // If city is not provided, prioritize interaction-based matches for best_coaches
    //         foreach ($allCoaches as $coach) {
    //             if ($interactionCoachIds->contains($coach->coach_id)) {
    //                 // Add to best_coaches if there's an interaction match
    //                 $bestCoaches->push($coach);
    //             } else {
    //                 // Otherwise, add to top_picks
    //                 $topPicks->push($coach);
    //             }
    //         }

    //         // Ensure both lists are balanced
    //         if ($bestCoaches->count() > $halfLimit) {
    //             $excess = $bestCoaches->count() - $halfLimit;
    //             $topPicks = $topPicks->merge($bestCoaches->splice($halfLimit, $excess));
    //         } elseif ($topPicks->count() > $halfLimit) {
    //             $excess = $topPicks->count() - $halfLimit;
    //             $bestCoaches = $bestCoaches->merge($topPicks->splice($halfLimit, $excess));
    //         }

    //     }

    //     // Add remaining coaches to balance the lists
    //     $remainingCoaches = $allCoaches->diff($bestCoaches)->diff($topPicks);

    //     foreach ($remainingCoaches as $coach) {
    //         if ($topPicks->count() < $halfLimit) {
    //             $topPicks->push($coach);
    //         } else {
    //             $bestCoaches->push($coach);
    //         }
    //     }

    //     // Sort Top Picks
    //     $topPicks = $topPicks->sortByDesc(function ($coach) {
    //         return [$coach->favorites_count, $coach->recent_interactions_count, $coach->check_in_details_count];
    //     })->values();

    //     // Sort Best Coaches
    //     $bestCoaches = $bestCoaches->sortByDesc(function ($coach) {
    //         return [$coach->favorites_count, $coach->recent_interactions_count, $coach->check_in_details_count];
    //     })->values();

    //     // Return response
    //     return response()->json([
    //         'status' => 'success',
    //         'message' => 'Coaches fetched successfully',
    //         'data' => [
    //             'top_picks' => $topPicks,
    //             'best_coaches' => $bestCoaches,
    //         ],
    //     ]);
    // }

    public function getAllCoachesClient(Request $request)
    {
        // Extract the authentication token
        $token = $request->bearerToken();
        if (!$token) {
            return response()->json(['status' => 'error', 'message' => 'Token not provided'], 401);
        }

        // Validate and decrypt token
        try {
            $decryptedToken = decrypt($token);
            [$clientId, $userType] = explode('-', $decryptedToken);

            if ($userType !== 'client') {
                return response()->json(['status' => 'error', 'message' => 'Invalid user type'], 401);
            }
        } catch (\Exception $e) {
            return response()->json(['status' => 'error', 'message' => 'Invalid token format'], 400);
        }

        // Make the city parameter optional
        $validator = Validator::make($request->all(), [
            'city' => 'nullable|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'message' => $validator->errors()->first()], 400);
        }

        $city = strtolower($request->city);

        // Fetch all coaches
        $allCoaches = Coach::query()
            ->withCount(['favorites', 'checkInDetails', 'recentInteractions']) // Include engagement counts
            ->with('activeClientPreferences')
            ->get();

        $bestCoaches = collect();
        $topPicks = collect();

        // Define limits for balanced distribution
        $totalCoaches = $allCoaches->count();
        $halfLimit = floor($totalCoaches / 2);

        // If city is provided, filter coaches by city
        if ($city) {
            // Helper function to fetch nearby cities based on postal code
            $nearbyCities = $this->getNearbyCities($city);

            // Separate coaches into Top Picks and Best Coaches based on city
            foreach ($allCoaches as $coach) {
                if (
                    str_contains(strtolower($coach->city), $city) && // Exact city match
                    $topPicks->count() < $halfLimit // Ensure Top Picks doesn't exceed the limit
                ) {
                    $topPicks->push($coach);
                } elseif (
                    $bestCoaches->count() < $halfLimit // Ensure Best Coaches doesn't exceed the limit
                ) {
                    $bestCoaches->push($coach);
                } elseif (
                    in_array($coach->city, $nearbyCities) && // Nearby city match
                    $topPicks->count() < $halfLimit // Ensure Top Picks doesn't exceed the limit
                ) {
                    $topPicks->push($coach);
                }
            }

        } else {
            // If city is not provided, prioritize interaction-based matches for best_coaches
            foreach ($allCoaches as $coach) {
                if ($bestCoaches->count() < $halfLimit) {
                    $bestCoaches->push($coach);
                } else {
                    $topPicks->push($coach);
                }
            }

            // Ensure both lists are balanced
            if ($bestCoaches->count() > $halfLimit) {
                $excess = $bestCoaches->count() - $halfLimit;
                $topPicks = $topPicks->merge($bestCoaches->splice($halfLimit, $excess));
            } elseif ($topPicks->count() > $halfLimit) {
                $excess = $topPicks->count() - $halfLimit;
                $bestCoaches = $bestCoaches->merge($topPicks->splice($halfLimit, $excess));
            }
        }

        // Add remaining coaches to balance the lists
        $remainingCoaches = $allCoaches->diff($bestCoaches)->diff($topPicks);

        foreach ($remainingCoaches as $coach) {
            if ($topPicks->count() < $halfLimit) {
                $topPicks->push($coach);
            } else {
                $bestCoaches->push($coach);
            }
        }

        // Add isFav and isPref keys to the coach data for both top_picks and best_coaches
        foreach ([$topPicks, $bestCoaches] as &$coaches) {
            foreach ($coaches as &$coach) {
                // Check if the client has marked this coach as a favorite
                $coach->isFav = FavoriteCoaches::where('client_id', $clientId)
                    ->where('coach_id', $coach->coach_id)->exists();

                // Check if the client has added this coach to their preferences
                $preference = clientPreferences::where('client_id', $clientId)
                    ->where('coach_id', $coach->coach_id)
                    ->first();

                if ($preference) {
                    // Check the status of the preference
                    if (
                        ($preference->status_by_client === 'rejected' || $preference->status_by_coach === 'rejected') ||
                        ($preference->status_by_client === 'accepted' && $preference->status_by_coach === 'rejected') ||
                        ($preference->status_by_coach === 'accepted' && $preference->status_by_client === 'rejected')
                    ) {
                        $coach->isPref = false;
                    } else {
                        $coach->isPref = true;
                    }
                } else {
                    $coach->isPref = true; // Default to true if no preference exists
                }
            }
        }

        // Return response
        return response()->json([
            'status' => 'success',
            'message' => 'Coaches fetched successfully',
            'data' => [
                'top_picks' => $topPicks,
                'best_coaches' => $bestCoaches,
            ],
        ]);
    }




    private function getNearbyCities($city)
    {
        // Helper function to fetch nearby cities based on postal code proximity.
        // Fetch the postal code for the given city
        $postalCode = Coach::whereRaw('LOWER(city) LIKE ?', ['%' . $city . '%'])
            ->value('postal_code');

        if (!$postalCode) {
            return []; // Return empty array if postal code not found
        }

        // Fetch nearby postal codes (this can be refined based on a proper proximity algorithm)
        $nearbyPostalCodes = Coach::where('postal_code', 'LIKE', substr($postalCode, 0, 3) . '%')->pluck('city');

        return $nearbyPostalCodes->map(function ($city) {
            return strtolower($city);
        })->toArray();
    }






    public function getAllCoachesWithoutFilter()
    {
        // Fetch all coaches from the database
        $coaches = Coach::with('activeClientPreferences')->get();

        // Check if coaches exist
        if ($coaches->isEmpty()) {
            return response()->json([
                'status' => 'success',
                'message' => 'No coaches found',
                'coaches' => $coaches,
            ], 200);
        }

        // Return the list of all coaches
        return response()->json([
            'status' => 'success',
            'message' => 'Coaches fetched successfully',
            'coaches' => $coaches,
        ]);
    }


    public function getWeeklyProgress(Request $request)
    {
        // Extract the authentication token
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract client_id
        try {
            $decryptedToken = decrypt($token);
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;

            // Validate the user_id and user_type
            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Fetch the weekly progress details for the client
        $checkInDetails = checkInDetails::where('client_id', $clientId)
                                        ->with('checkInPictures') // Assuming you have a relationship for pictures
                                        ->get();

        // Check if any check-in details exist for the client
        if ($checkInDetails->isEmpty()) {
            return response()->json([
                'status' => 'success', 
                'message' => 'No progress found for this week',
                'details' => $checkInDetails,
            ], 200);
        }

        // Return the weekly progress details
        return response()->json([
            'status' => 'success',
            'message' => 'Weekly progress fetched successfully',
            'details' => $checkInDetails,
        ]);
    }




    public function addWeeklyProgress(Request $request)
    {
        // Extract the authentication token
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract client_id
        try {
            $decryptedToken = decrypt($token);
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;

            // Validate the user_id and user_type
            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Validate the input
        $validator = Validator::make($request->all(), [
            'check_in_pictures' => 'required|array|min:1',
            'check_in_pictures.*' => 'file|image|mimes:jpeg,png,jpg,gif',
            'chest' => 'nullable',
            'shoulders' => 'nullable',
            'waist' => 'nullable',
            'hips' => 'nullable',
            'thighs' => 'nullable',
            'arms' => 'nullable',
            'depression' => 'nullable',
            'macros' => 'nullable',
            'missed_meals' => 'nullable',
            'cheat_meals' => 'nullable',
            'sleep' => 'nullable',
            'appetite_and_digestion' => 'nullable',
            'recovery' => 'nullable',
            'training_strength' => 'nullable',
            'coach_feedback' => 'nullable',
        ]); 

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first()
            ], 400);
        }

        // Fetch the coach_id from the Clients table based on client_id
        $client = Clients::find($clientId);
        if (!$client) {
            return response()->json([
                'status' => 'error',
                'message' => 'Client not found'
            ], 404);
        }

        // Now we have the coach_id from the client
        $coachId = $client->coach_id;

        // Check if coach_id is null or invalid
        if (!$coachId) {
            return response()->json([
                'status' => 'error',
                'message' => 'Coach not found for this client'
            ], 404);
        }

        // Store check-in details
        $checkInDetails = new CheckInDetails();
        $checkInDetails->client_id = $clientId;
        $checkInDetails->coach_id = $coachId; // Set the coach_id from the client's data

        // Always store date-related fields
        $checkInDetails->checkin_date = Carbon::now()->toDateString();
        $checkInDetails->checkin_day = Carbon::now()->format('l');
        $checkInDetails->checkin_week = Carbon::now()->week;
        $checkInDetails->checkin_month = Carbon::now()->month;
        $checkInDetails->checkin_year = Carbon::now()->year;
        $checkInDetails->checkin_time = Carbon::now()->toTimeString();

        // Store optional fields only if they are provided
        $checkInDetails->chest = $request->chest ?? null;
        $checkInDetails->shoulders = $request->shoulders ?? null;
        $checkInDetails->waist = $request->waist ?? null;
        $checkInDetails->hips = $request->hips ?? null;
        $checkInDetails->thighs = $request->thighs ?? null;
        $checkInDetails->arms = $request->arms ?? null;
        $checkInDetails->how_was_your_week = $request->week ?? null;
        $checkInDetails->any_stress_depression = $request->depression ?? null;
        $checkInDetails->how_was_macros = $request->macros ?? null;
        $checkInDetails->how_many_missed_meals = $request->missed_meals ?? null;
        $checkInDetails->how_many_cheat_meals = $request->cheat_meals ?? null;
        $checkInDetails->how_was_sleep = $request->sleep ?? null;
        $checkInDetails->how_is_appetite_and_digestion = $request->appetite_and_digestion ?? null;
        $checkInDetails->how_is_recovery = $request->recovery ?? null;
        $checkInDetails->training_strength = $request->training_strength ?? null;
        $checkInDetails->coach_feedback = $request->coach_feedback ?? null;

        // Save the check-in details
        $checkInDetails->save();

        // Handle file uploads for check_in_pictures
        if ($request->hasFile('check_in_pictures')) {
            foreach ($request->file('check_in_pictures') as $file) {
                if ($file->isValid()) {
                    // Get the original file name
                    $filename = $file->getClientOriginalName(); // Get the original file name
                    
                    // Give a custom file name (you can replace "check_in_picture" with whatever custom name you prefer)
                    $getfilenamewitoutext = "check_in_picture"; // Custom file name (just like profile_picture)
                    $getfileExtension = $file->getClientOriginalExtension(); // Get the file extension
                    $createnewFileName = str_replace(' ', '_', $getfilenamewitoutext) . '_' . time() . '.' . $getfileExtension; // Create a new random file name

                    // Store the file in the 'public/check_in_pictures' directory
                    $img_path = $file->storeAs('public/check_in_pictures', $createnewFileName); // Store and get the image path

                    // Store the new file path in the database
                    $checkInPicture = new CheckInPictures();
                    $checkInPicture->picture = '/storage/check_in_pictures/' . $createnewFileName; // Store the path in DB
                    $checkInPicture->check_in_details_id = $checkInDetails->details_id; // Reference to check_in_details
                    $checkInPicture->save();
                }
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Weekly progress added successfully',
            'check_in_details' => $checkInDetails
        ], 200);
    }





    // public function clientHome(Request $request)
    // {
    //     // Extract the authentication token
    //     $token = $request->bearerToken();

    //     if (!$token) {
    //         return response()->json([
    //             'message' => 'Token not provided',
    //             'has_coach' => false,
    //             'data' => null,
    //         ], 401);
    //     }

    //     // Retrieve the session based on the provided authentication token
    //     $session = Session::where('authentication_token', $token)->first();

    //     if (!$session) {
    //         return response()->json([
    //             'message' => 'Invalid token or session not found',
    //             'has_coach' => false,
    //             'data' => null,
    //         ], 401);
    //     }

    //     // Decrypt the authentication token to extract client_id
    //     try {
    //         $decryptedToken = decrypt($token);
    //         [$userId, $userType] = explode('-', $decryptedToken);

    //         // Validate the user_id and user_type
    //         if ($session->user_id != $userId || $userType !== 'client') {
    //             return response()->json([
    //                 'message' => 'Token does not match the session or user type is not client',
    //                 'has_coach' => false,
    //                 'data' => null,
    //             ], 401);
    //         }
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'message' => 'Invalid token format',
    //             'has_coach' => false,
    //             'data' => null,
    //         ], 400);
    //     }



    //     // Validate the incoming request
    //     $validator = Validator::make($request->all(), [
    //         'city' => 'nullable|max:255',
    //     ]);

    //     if ($validator->fails()) {
    //         return response()->json(['error' => $validator->errors()->first()], 400);
    //     }


    //     // Retrieve the client using the extracted client_id
    //     $client = Clients::find($userId);

    //     if (!$client) {
    //         return response()->json([
    //             'message' => 'Client not found',
    //             'has_coach' => false,
    //             'data' => null,
    //         ], 404);
    //     }

    //     // Check if the client has a coach
    //     $hasCoach = !empty($client->coach_id);

    //     if ($hasCoach) {
    //         // If the client has a coach, fetch all assigned plans
    //         $assignedPlans = $this->fetchAllAssignedPlans($request);
    //         $responseData = $assignedPlans->getData();
    //         $data = $responseData->data;
    //     } else {
    //         // If the client does not have a coach, fetch all available coaches
    //         $coachesData = $this->getAllCoachesClient($request);
    //         $responseData = $coachesData->getData();
    //         $data = $responseData->data;
    //     }

    //     // Return the response in the required format
    //     return response()->json([
    //         'status' => 'Success',
    //         'has_coach' => $hasCoach,
    //         'data' => $data,
    //     ], 200);
    // }

    public function clientHome(Request $request)
    {
        // Extract the authentication token
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'message' => 'Token not provided',
                'has_coach' => false,
                'data' => null,
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'message' => 'Invalid token or session not found',
                'has_coach' => false,
                'data' => null,
            ], 401);
        }

        // Decrypt the authentication token to extract client_id
        try {
            $decryptedToken = decrypt($token);
            [$userId, $userType] = explode('-', $decryptedToken);

            // Validate the user_id and user_type
            if ($session->user_id != $userId || $userType !== 'client') {
                return response()->json([
                    'message' => 'Token does not match the session or user type is not client',
                    'has_coach' => false,
                    'data' => null,
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Invalid token format',
                'has_coach' => false,
                'data' => null,
            ], 400);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'city' => 'nullable|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Retrieve the client using the extracted client_id
        $client = Clients::find($userId);

        if (!$client) {
            return response()->json([
                'message' => 'Client not found',
                'has_coach' => false,
                'data' => null,
            ], 404);
        }

        // Check if the client has a coach
        $hasCoach = !empty($client->coach_id);

        if ($hasCoach) {
            // If the client has a coach, fetch all assigned plans
            $assignedPlans = $this->fetchAllAssignedPlans($request);
            $responseData = $assignedPlans->getData();
            $data = (array) $responseData->data; // Cast data to an array

        } else {
            // If the client does not have a coach, fetch all available coaches
            $coachesData = $this->getAllCoachesClient($request);
            $responseData = $coachesData->getData();
            $data = (array) $responseData->data; // Cast data to an array
        }

        // Add isFav and isPref keys to the coach data for both top_picks and best_coaches
        foreach (['top_picks', 'best_coaches'] as $key) {
            // Check if data is an object and the key exists
            if (isset($data[$key]) && is_array($data[$key])) {              
                foreach ($data[$key] as &$coach) {
                    // Check if the client has marked this coach as a favorite
                    $coach->isFav = FavoriteCoaches::where('client_id', $client->client_id)
                        ->where('coach_id', $coach->coach_id)->exists();

                    // Check if the client has added this coach to their preferences
                    $preference = clientPreferences::where('client_id', $client->client_id)
                        ->where('coach_id', $coach->coach_id)
                        ->first();

                    if ($preference) {
                        // Check the status of the preference
                        if (
                            ($preference->status_by_client === 'rejected' || $preference->status_by_coach === 'rejected') ||
                            ($preference->status_by_client === 'accepted' && $preference->status_by_coach === 'rejected') ||
                            ($preference->status_by_coach === 'accepted' && $preference->status_by_client === 'rejected')
                        ) {
                            $coach->isPref = false;
                        } else {
                            $coach->isPref = true;
                        }
                    } else {
                        $coach->isPref = true; // Default to true if no preference exists
                    }
                }
            }
        }



        // Return the response in the required format
        return response()->json([
            'status' => 'Success',
            'has_coach' => $hasCoach,
            'data' => $data,
        ], 200);
    }






    public function searchCoaches(Request $request)
    {
        // Extract the authentication token
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token);
            [$userId, $userType] = explode('-', $decryptedToken);

            // Validate the user_id and user_type
            if ($session->user_id != $userId || $userType !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'keyword' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first(),
            ], 400);
        }

        // Extract the keyword from the request
        $keyword = $request->input('keyword');

        // Search for coaches based on the keyword
        $coaches = Coach::where('full_name', 'like', "%$keyword%")
            ->orWhere('username', 'like', "%$keyword%")
            ->orWhere('city', 'like', "%$keyword%")
            ->orWhere('email', 'like', "%$keyword%")
            // ->orWhere('bio', 'like', "%$keyword%")
            ->with('activeClientPreferences')
            ->get();


        // Add isFav and isPref to each coach in the recent coaches list
        foreach ($coaches as &$coach) {
            // Check if the client has marked this coach as a favorite
            $coach->isFav = FavoriteCoaches::where('client_id', $userId)
                ->where('coach_id', $coach->coach_id)
                ->exists();

            // Check if the client has added this coach to their preferences
            $preference = clientPreferences::where('client_id', $userId)
                ->where('coach_id', $coach->coach_id)
                ->first();

            if ($preference) {
                // Check the status of the preference
                if (
                    ($preference->status_by_client === 'rejected' || $preference->status_by_coach === 'rejected') ||
                    ($preference->status_by_client === 'accepted' && $preference->status_by_coach === 'rejected') ||
                    ($preference->status_by_coach === 'accepted' && $preference->status_by_client === 'rejected')
                ) {
                    $coach->isPref = false;
                } else {
                    $coach->isPref = true;
                }
            } else {
                $coach->isPref = true; // Default to true if no preference exists
            }
        }


        // Return the response
        return response()->json([
            'status' => 'success',
            'message' => 'Coaches fetched successfully',
            'data' => $coaches,
        ], 200);
    }




    public function cancelCoachPreference(Request $request)
    {
        // Extract the authentication token from the Authorization header
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'status' => 'error',
                'message' => 'Token not provided',
            ], 401);
        }

        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'coach_id' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->first()], 400);
        }

        // Attempt to retrieve the coach by coach_id
        $coach = Coach::where('coach_id', $request->coach_id)->first();

        // Check if the coach exists in the database
        if (!$coach) {
            return $this->errorResponse('No coach found', self::STATUS_NOT_FOUND);
        }

        // Retrieve the session based on the provided authentication token
        $session = Session::where('authentication_token', $token)->first();

        if (!$session) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token or session not found',
            ], 401);
        }

        // Decrypt the authentication token to extract user_id and user_type
        try {
            $decryptedToken = decrypt($token); // Example: "123-client"
            [$clientId, $userType] = explode('-', $decryptedToken);

            // or we can just use this to retrieve the client_id from the session:
            // $clientId = $session->user_id;


            if ($session->user_id != $clientId || $session->user_type !== 'client') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token does not match the session or user type is not client',
                ], 401);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid token format',
            ], 400);
        }

        // Check if the client exists
        $client = Clients::find($clientId);
        if (!$client) {
            return response()->json([
                'status' => 'error',
                'message' => 'Client not found',
            ], 404);
        }

        // Fetch the client preference from the database
        $client_preference = clientPreferences::where('client_id', $clientId)->where('coach_id', $request->coach_id)->first();
        if (!$client_preference) {
            return $this->errorResponse('Client preference not found', self::STATUS_NOT_FOUND);
        }

        $client_preference->client_status = 'cancel';
        $client_preference->message = 'cancelled by client';
        $client_preference->active = 0;

        $updateSuccess = $client_preference->save(); // Save the updated password

        if ($updateSuccess) {
            return response()->json([
                'status' => 'success',
                'message' => 'Client preferences updated successfully',
            ], 200);

        } else {
            return $this->errorResponse('Failed to update client preferences', self::STATUS_INTERNAL_SERVER_ERROR);
        }


    }











}






