<?php

namespace App\Http\Controllers;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

use App\Models\User;
use App\Models\Coach;
use App\Models\Clients;
use App\Models\Session;
use App\Models\Subscription;
use App\Models\paymentMethod;
use App\Models\Invoice;

use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Carbon;

use Illuminate\Support\Facades\Storage;
use Stripe\StripeClient;
use Stripe\Event;
use Stripe\Webhook;
use Stripe\Exception\SignatureVerificationException;



use Illuminate\Support\Facades\Crypt;

class APIWebhookController extends Controller
{

    public function handleWebhook(Request $request)
    {
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $secret = env('STRIPE_WEBHOOK_SECRET'); 

        try {
            $event = Webhook::constructEvent($payload, $sigHeader, $secret);

            // Handle different event types
            switch ($event->type) {
                case 'customer.subscription.deleted':
                case 'customer.subscription.updated':
                case 'customer.subscription.paused':
                case 'customer.subscription.resumed':
                    $this->handleSubscriptionEvent($event);
                    break;

                case 'payment_method.attached':
                    $this->handlePaymentMethodAttached($event);
                    break;

                case 'invoice.paid':
                case 'invoice.payment_action_required':
                case 'invoice.payment_failed':
                    $this->handleInvoiceEvent($event);
                    break;

                // case 'invoice.created':
                //     $this->handleInvoiceCreated($event);
                //     break;

                default:
                    // Handle other events or log unexpected events
                    break;
            }

            return response()->json(['status' => 'success']);
        } catch (SignatureVerificationException $e) {
            return response()->json(['error' => 'Webhook signature verification failed.'], 400);
        } catch (\UnexpectedValueException $e) {
            return response()->json(['error' => 'Webhook payload parsing failed.'], 400);
        }
    }

    protected function handleSubscriptionEvent($event)
    {
        // Handle subscription events: update subscription status
        $subscription = $event->data->object;
        $customer = $subscription->customer;
        $stripeSubscriptionId = $subscription->id;

        // Update subscription status in your database
        Subscription::where('customer', $customer)
                    ->where('stripe_subscription_id', $stripeSubscriptionId)
                    ->update(['status' => $subscription->status]);
    }

    protected function handlePaymentMethodAttached($event)
    {
        // Handle payment method attached event: store or update payment method details
        $paymentMethod = $event->data->object;
        $customer = $paymentMethod->customer;
        $coachId = Coach::where('stripe_id', $customer)->value('coach_id');

        // Check if the payment method already exists for the coach
        $existingPaymentMethod = PaymentMethod::where('coach_id', $coachId)->first();

        if ($existingPaymentMethod) {
            // Update the existing payment method
            $existingPaymentMethod->update([
                'stripe_payment_method_id' => $paymentMethod->id,
                'type' => $paymentMethod->type,
                'card_brand' => $paymentMethod->card->brand,
                'last_4_digits' => $paymentMethod->card->last4,
                'exp_month' => $paymentMethod->card->exp_month,
                'exp_year' => $paymentMethod->card->exp_year,
                'country' => $paymentMethod->billing_details->address->country,
                'postal_code' => $paymentMethod->billing_details->address->postal_code,
            ]);
        } else {
            // Create a new payment method
            PaymentMethod::create([
                'stripe_payment_method_id' => $paymentMethod->id,
                'customer' => $customer,
                'coach_id' => $coachId,
                'type' => $paymentMethod->type,
                'card_brand' => $paymentMethod->card->brand,
                'last_4_digits' => $paymentMethod->card->last4,
                'exp_month' => $paymentMethod->card->exp_month,
                'exp_year' => $paymentMethod->card->exp_year,
                'country' => $paymentMethod->billing_details->address->country,
                'postal_code' => $paymentMethod->billing_details->address->postal_code,
            ]);
        }
    }


    protected function handleInvoiceEvent($event)
    {
        // Handle invoice events: update invoice status
        $invoice = $event->data->object;
        $customer = $invoice->customer;
        $stripeInvoiceId = $invoice->id;

        // Update invoice status in your database
        Invoice::where('customer', $customer)
               ->where('stripe_invoice_id', $stripeInvoiceId)
               ->update(['status' => $invoice->status]);
    }


    protected function handleInvoiceCreated($event)
    {
        // Handle invoice created event: store new invoice details
        $invoice = $event->data->object;

        // Store invoice details in invoices table
        Invoice::create([
            'stripe_invoice_id' => $invoice->id,
            'customer' => $invoice->customer,
            'coach_id' => Coach::where('stripe_id', $invoice->customer)->value('coach_id'),
            'amount_paid' => $invoice->amount_paid / 100, // Convert to dollars
            'status' => $invoice->status,
        ]);

        $stripe = new StripeClient(env('STRIPE_SECRET_KEY'));
        // Call Stripe API to update invoice to auto-advance
        $stripe->invoices->update($invoice->id, ['auto_advance' => true]);

    }

}





